/**
 * @file main.c
 * @brief Main function is where the program starts execution
 * @created 1999-08-17
 * @date 2007-08-30
 */
/*
 * copyright (c) 1998-2007 TLK Games all rights reserved
 * $Id: main.c,v 1.10 2007/08/30 20:17:18 gurumeditation Exp $
 *
 * Powermanga is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Powermanga is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#include <stdbool.h>
#ifdef _WIN32
#include <windows.h>
/* for ShellExecute, to launch HTLM help */
#include "shellapi.h"
#endif
#include "powermanga.h"
#include "tools.h"
#include "images.h"
#include "config_file.h"
#include "curve_phase.h"
#include "display.h"
#include "electrical_shock.h"
#include "enemies.h"
#include "bonus.h"
#include "energy_gauge.h"
#include "explosions.h"
#include "shots.h"
#include "extra_gun.h"
#include "gfx_wrapper.h"
#include "guardians.h"
#include "menu.h"
#include "meteors_phase.h"
#include "movie.h"
#include "options_panel.h"
#include "scrolltext.h"
#include "satellite_protections.h"
#include "script_page.h"
#include "shockwave.h"
#include "spaceship.h"
#include "sprites_string.h"
#include "sdl_mixer.h"
#include "starfield.h"
#include "texts.h"
#include "text_overlay.h"
#include <errno.h>

#ifdef SHAREWARE_VERSION
#include <SDL/SDL_ttf.h>
#include "counter_shareware.h"
#endif

/* TRUE = leave the Powermanga game */
bool quit_game = FALSE;
#ifdef POWERMANGA_SDL
/* game speed : 70 frames/sec (1000 <=> 1 seconde ; 1000 / 70 =~ 14) */
static const Uint32 GAME_FRAME_RATE = 14;
/* movie speed: 28 frames/sec */
static const Uint32 MOVIE_FRAME_RATE = 35;
#else
/* game speed : 70 frames/sec (1000000 <=> 1 seconde ; 1000000 / 70 =~ 14286) */
static const Uint32 GAME_FRAME_RATE = 14286;
/* movie speed: 28 frames/sec */
Uint32 MOVIE_FRAME_RATE = 35715;
#endif

static bool initialize_and_run (void);
static void main_loop (void);

/**
 * Returns to the standard GP2X menu.
 */
#ifdef POWERMANGA_GP2X
void
returnToMenu (void)
{
  /* This is how to quit back to the menu - calling exit() will just cause
   * the GP2X to "hang". execl() will replace the current process image
   *  with that of the menu program, and then execute it */
  chdir ("/usr/gp2x");
  execl ("/usr/gp2x/gp2xmenu", "/usr/gp2x/gp2xmenu", NULL);
}
#else
#ifdef POWERMANGA_PSP
void
returnToMenu (void)
{
  sceKernelExitGame ();
}
#endif
#endif

/**
 * The main function is where the program starts execution.
 */
Sint32
main (Sint32 args_count, char **arguments)
{
  Sint32 verbose;
  /* GP2X or PSP port */
#ifdef POWERMANGA_HANDHELD_CONSOLE
  /* Use atexit() to call the return-to-menu function,
   * in case of crashes, etc. */
  atexit (returnToMenu);
#endif

  /* allocate memory table */
  if (!memory_init (22000))
    {
      exit (0);
    }

  /* load config file */
  if (!configfile_load ())
    {
      memory_releases_all (0);
      exit (0);
    }
  verbose = power_conf->verbose;
  if (configfile_scan_arguments (args_count, arguments))
    {
#ifdef POWERMANGA_HANDHELD_CONSOLE
      /* We require a 320x200 output size to fit on
       * the GP2X or PSP's display */
      power_conf->scale_x = 1;
      power_conf->resolution = 320;
      power_conf->fullscreen = 1;
      pixel_size = 1;
#endif
      if (power_conf->resolution == 320)
        {
          vmode = 0;
        }
      else
        {
          if (power_conf->scale_x > 1)
            {
              vmode = 2;
            }
          else
            {
              vmode = 1;
            }
        }
      initialize_and_run ();
    }
  release_game ();

  /* releases all memory allocated */
  memory_releases_all (verbose);

  /* launch html donation page before leaving */
#ifdef _WIN32
  ShellExecute (0, "open", ".\\html\\ar01s06s02.html", 0, 0, SW_SHOWNORMAL);
#endif

  exit (0);
}

/**
 * Initialize and run the game!
 * @return TRUE if it completed successfully or FALSE otherwise
 */
static bool
initialize_and_run (void)
{
  Sint32 verbose = power_conf->verbose;
  if (verbose)
    {
      configfile_print ();
    }

#ifdef VERBOSE
  if (verbose)
    {
      fprintf (stdout, "*********************************************"
               "************************************\n"
               "> main.c/powermanga() : " POWERMANGA_VERSION "\n");
      type_routine_gfx ();
    }
#endif
  if (!inits_game ())
    {
      return FALSE;
    }

#ifdef SHAREWARE_VERSION
  /* update counter */
  Sint32 cpt = counter_shareware_update ("PowerManga", 0, 8);
  if (cpt >= 8)
    {
      /* display first page order */
      switch (power_conf->lang)
        {
        case FR_LANG:
          show_page_order (1, "fr", cpt);
          break;
        case EN_LANG:
          show_page_order (1, "en", cpt);
          break;
        default:
          show_page_order (1, "en", cpt);
          break;
        }
    }
  counter_shareware_free ();
#endif

  fps_init ();
  main_loop ();
  fps_print ();

#ifdef SHAREWARE_VERSION
  /* displaying of the third page order */
  switch (power_conf->lang)
    {
    case FR_LANG:
      show_page_order (3, "fr", 0);
      break;
    case EN_LANG:
      show_page_order (3, "en", 0);
      break;
    default:
      show_page_order (3, "en", 0);
      break;
    }
  TTF_Quit ();
#endif

#ifdef VERBOSE
  if (verbose > 0)
    {
      fprintf (stdout, "> main.c/powermanga(): The End\n");
    }
#endif
  return TRUE;
}

/**
 * Main loop of the Powermanga game
 */
void
main_loop (void)
{
  Sint32 pause_delay = 0;
  Sint32 frame_diff = 0;
  do
    {
      loops_counter++;
      if (!power_conf->nosync)
        {
          frame_diff = get_time_difference ();
          if (movie_playing_switch != MOVIE_NOT_PLAYED)
            {
              pause_delay =
                wait_next_frame (MOVIE_FRAME_RATE - frame_diff + pause_delay,
                                 MOVIE_FRAME_RATE);
            }
          else
            {
              pause_delay =
                wait_next_frame (GAME_FRAME_RATE - frame_diff + pause_delay,
                                 GAME_FRAME_RATE);
            }
        }
      /* handle Powermanga game */
      if (!update_frame ())
        {
          quit_game = TRUE;
        }
      /* handle keyboard and joystick events */
      display_handle_events ();

      /* update our main window */
      display_update_window ();

#ifdef USE_SDLMIXER
      /* play music and sounds */
      sound_handle ();
#endif
    }
  while (!quit_game);
}


/*
 * affich_white      is_mask
 * pow_of_dest
 * etat_dommages     energy_level
 * cad_tir_initiale  fire_rate
 * cad_tir           fire_rate_count
 * nbr_tot_vbl       anim_speed
 * nbr_vbl_act       anim_count
 * img_act           current_image
 * nbr_img           numof_images
 * */
