#
# Copyright (c) 2012 The Georgia Institute of Technology
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation;
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Author: Brian Swenson <bswenson3@gatech.edu>
# Modified by: Gabriel Ferreira <gabrielcarvfer@gmail.com>
#

try:
    from ns import ns
except ModuleNotFoundError:
    raise SystemExit(
        "Error: ns3 Python module not found;"
        " Python bindings may not be enabled"
        " or your PYTHONPATH might not be properly configured"
    )

ns.LogComponentEnable("BriteTopologyHelper", ns.LOG_LEVEL_ALL)

# BRITE needs a configuration file to build its graph. By default, this
# example will use the TD_ASBarabasi_RTWaxman.conf file. There are many others
# which can be found in the BRITE/conf_files directory
confFile = "src/brite/examples/conf_files/TD_ASBarabasi_RTWaxman.conf"

# Invoke the BriteTopologyHelper and pass in a BRITE
# configuration file and a seed file. This will use
# BRITE to build a graph from which we can build the ns-3 topology
bth = ns.BriteTopologyHelper(confFile)
bth.AssignStreams(3)

p2p = ns.PointToPointHelper()

stack = ns.InternetStackHelper()

nixRouting = ns.Ipv4NixVectorHelper()
stack.SetRoutingHelper(nixRouting)

address = ns.Ipv4AddressHelper()
address.SetBase("10.0.0.0", "255.255.255.252")

bth.BuildBriteTopology(stack)
bth.AssignIpv4Addresses(address)

print(f"Number of AS created {bth.GetNAs()}")

# The BRITE topology generator generates a topology of routers.  Here we create
# two subnetworks which we attach to router leaf nodes generated by BRITE
# use just one node
client = ns.NodeContainer()
server = ns.NodeContainer()

client.Create(1)
stack.Install(client)

# install client node on last leaf node of AS 0
numLeafNodesInAsZero = bth.GetNLeafNodesForAs(0)
client.Add(bth.GetLeafNodeForAs(0, numLeafNodesInAsZero - 1))

server.Create(1)
stack.Install(server)

# install server node on last leaf node on AS 1
numLeafNodesInAsOne = bth.GetNLeafNodesForAs(1)
server.Add(bth.GetLeafNodeForAs(1, numLeafNodesInAsOne - 1))

p2p.SetDeviceAttribute("DataRate", ns.StringValue("5Mbps"))
p2p.SetChannelAttribute("Delay", ns.StringValue("2ms"))

p2pClientDevices = p2p.Install(client)
p2pServerDevices = p2p.Install(server)

address.SetBase("10.1.0.0", "255.255.0.0")
clientInterfaces = address.Assign(p2pClientDevices)

address.SetBase("10.2.0.0", "255.255.0.0")
serverInterfaces = ns.Ipv4InterfaceContainer()
serverInterfaces = address.Assign(p2pServerDevices)

echoServer = ns.UdpEchoServerHelper(9)
serverApps = echoServer.Install(server.Get(0))
serverApps.Start(ns.Seconds(1.0))
serverApps.Stop(ns.Seconds(5.0))

echoClient = ns.UdpEchoClientHelper(serverInterfaces.GetAddress(0).ConvertTo(), 9)
echoClient.SetAttribute("MaxPackets", ns.UintegerValue(1))
echoClient.SetAttribute("Interval", ns.TimeValue(ns.Seconds(1.0)))
echoClient.SetAttribute("PacketSize", ns.UintegerValue(1024))

clientApps = echoClient.Install(client.Get(0))
clientApps.Start(ns.Seconds(2.0))
clientApps.Stop(ns.Seconds(5.0))

asciiTrace = ns.AsciiTraceHelper()
p2p.EnableAsciiAll(asciiTrace.CreateFileStream("briteLeaves.tr"))

# Run the simulator
ns.Simulator.Stop(ns.Seconds(6.0))
ns.Simulator.Run()
ns.Simulator.Destroy()
