/* -LICENSE-START-
** Copyright (c) 2009 Blackmagic Design
**
** Permission is hereby granted, free of charge, to any person or organization
** obtaining a copy of the software and accompanying documentation covered by
** this license (the "Software") to use, reproduce, display, distribute,
** execute, and transmit the Software, and to prepare derivative works of the
** Software, and to permit third-parties to whom the Software is furnished to
** do so, all subject to the following:
** 
** The copyright notices in the Software and this entire statement, including
** the above license grant, this restriction and the following disclaimer,
** must be included in all copies of the Software, in whole or in part, and
** all derivative works of the Software, unless such copies or derivative
** works are solely in the form of machine-executable object code generated by
** a source language processor.
** 
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
** -LICENSE-END-
*/

/* DeckLinkAPI.h */
		
#ifndef __DeckLink_API_h__
#define __DeckLink_API_h__
		
#include <stdint.h>
#include "LinuxCOM.h"

#define BLACKMAGIC_DECKLINK_API_MAGIC	1

// Type Declarations

typedef int64_t BMDTimeValue;
typedef int64_t BMDTimeScale;
typedef uint32_t BMDTimecodeBCD;
typedef uint32_t BMDTimecodeUserBits;


// Interface ID Declarations

#define IID_IDeckLinkVideoOutputCallback                 /* 20AA5225-1958-47CB-820B-80A8D521A6EE */ (REFIID){0x20,0xAA,0x52,0x25,0x19,0x58,0x47,0xCB,0x82,0x0B,0x80,0xA8,0xD5,0x21,0xA6,0xEE}
#define IID_IDeckLinkInputCallback                       /* DD04E5EC-7415-42AB-AE4A-E80C4DFC044A */ (REFIID){0xDD,0x04,0xE5,0xEC,0x74,0x15,0x42,0xAB,0xAE,0x4A,0xE8,0x0C,0x4D,0xFC,0x04,0x4A}
#define IID_IDeckLinkMemoryAllocator                     /* B36EB6E7-9D29-4AA8-92EF-843B87A289E8 */ (REFIID){0xB3,0x6E,0xB6,0xE7,0x9D,0x29,0x4A,0xA8,0x92,0xEF,0x84,0x3B,0x87,0xA2,0x89,0xE8}
#define IID_IDeckLinkAudioOutputCallback                 /* 403C681B-7F46-4A12-B993-2BB127084EE6 */ (REFIID){0x40,0x3C,0x68,0x1B,0x7F,0x46,0x4A,0x12,0xB9,0x93,0x2B,0xB1,0x27,0x08,0x4E,0xE6}
#define IID_IDeckLinkIterator                            /* 74E936FC-CC28-4A67-81A0-1E94E52D4E69 */ (REFIID){0x74,0xE9,0x36,0xFC,0xCC,0x28,0x4A,0x67,0x81,0xA0,0x1E,0x94,0xE5,0x2D,0x4E,0x69}
#define IID_IDeckLinkAPIInformation                      /* 7BEA3C68-730D-4322-AF34-8A7152B532A4 */ (REFIID){0x7B,0xEA,0x3C,0x68,0x73,0x0D,0x43,0x22,0xAF,0x34,0x8A,0x71,0x52,0xB5,0x32,0xA4}
#define IID_IDeckLinkDisplayModeIterator                 /* 9C88499F-F601-4021-B80B-032E4EB41C35 */ (REFIID){0x9C,0x88,0x49,0x9F,0xF6,0x01,0x40,0x21,0xB8,0x0B,0x03,0x2E,0x4E,0xB4,0x1C,0x35}
#define IID_IDeckLinkDisplayMode                         /* 3EB2C1AB-0A3D-4523-A3AD-F40D7FB14E78 */ (REFIID){0x3E,0xB2,0xC1,0xAB,0x0A,0x3D,0x45,0x23,0xA3,0xAD,0xF4,0x0D,0x7F,0xB1,0x4E,0x78}
#define IID_IDeckLink                                    /* 62BFF75D-6569-4E55-8D4D-66AA03829ABC */ (REFIID){0x62,0xBF,0xF7,0x5D,0x65,0x69,0x4E,0x55,0x8D,0x4D,0x66,0xAA,0x03,0x82,0x9A,0xBC}
#define IID_IDeckLinkOutput                              /* A3EF0963-0862-44ED-92A9-EE89ABF431C7 */ (REFIID){0xA3,0xEF,0x09,0x63,0x08,0x62,0x44,0xED,0x92,0xA9,0xEE,0x89,0xAB,0xF4,0x31,0xC7}
#define IID_IDeckLinkInput                               /* 6D40EF78-28B9-4E21-990D-95BB7750A04F */ (REFIID){0x6D,0x40,0xEF,0x78,0x28,0xB9,0x4E,0x21,0x99,0x0D,0x95,0xBB,0x77,0x50,0xA0,0x4F}
#define IID_IDeckLinkTimecode                            /* BC6CFBD3-8317-4325-AC1C-1216391E9340 */ (REFIID){0xBC,0x6C,0xFB,0xD3,0x83,0x17,0x43,0x25,0xAC,0x1C,0x12,0x16,0x39,0x1E,0x93,0x40}
#define IID_IDeckLinkVideoFrame                          /* 3F716FE0-F023-4111-BE5D-EF4414C05B17 */ (REFIID){0x3F,0x71,0x6F,0xE0,0xF0,0x23,0x41,0x11,0xBE,0x5D,0xEF,0x44,0x14,0xC0,0x5B,0x17}
#define IID_IDeckLinkMutableVideoFrame                   /* 69E2639F-40DA-4E19-B6F2-20ACE815C390 */ (REFIID){0x69,0xE2,0x63,0x9F,0x40,0xDA,0x4E,0x19,0xB6,0xF2,0x20,0xAC,0xE8,0x15,0xC3,0x90}
#define IID_IDeckLinkVideoFrame3DExtensions              /* DA0F7E4A-EDC7-48A8-9CDD-2DB51C729CD7 */ (REFIID){0xDA,0x0F,0x7E,0x4A,0xED,0xC7,0x48,0xA8,0x9C,0xDD,0x2D,0xB5,0x1C,0x72,0x9C,0xD7}
#define IID_IDeckLinkVideoInputFrame                     /* 05CFE374-537C-4094-9A57-680525118F44 */ (REFIID){0x05,0xCF,0xE3,0x74,0x53,0x7C,0x40,0x94,0x9A,0x57,0x68,0x05,0x25,0x11,0x8F,0x44}
#define IID_IDeckLinkVideoFrameAncillary                 /* 732E723C-D1A4-4E29-9E8E-4A88797A0004 */ (REFIID){0x73,0x2E,0x72,0x3C,0xD1,0xA4,0x4E,0x29,0x9E,0x8E,0x4A,0x88,0x79,0x7A,0x00,0x04}
#define IID_IDeckLinkAudioInputPacket                    /* E43D5870-2894-11DE-8C30-0800200C9A66 */ (REFIID){0xE4,0x3D,0x58,0x70,0x28,0x94,0x11,0xDE,0x8C,0x30,0x08,0x00,0x20,0x0C,0x9A,0x66}
#define IID_IDeckLinkScreenPreviewCallback               /* B1D3F49A-85FE-4C5D-95C8-0B5D5DCCD438 */ (REFIID){0xB1,0xD3,0xF4,0x9A,0x85,0xFE,0x4C,0x5D,0x95,0xC8,0x0B,0x5D,0x5D,0xCC,0xD4,0x38}
#define IID_IDeckLinkGLScreenPreviewHelper               /* 504E2209-CAC7-4C1A-9FB4-C5BB6274D22F */ (REFIID){0x50,0x4E,0x22,0x09,0xCA,0xC7,0x4C,0x1A,0x9F,0xB4,0xC5,0xBB,0x62,0x74,0xD2,0x2F}
#define IID_IDeckLinkConfiguration                       /* C679A35B-610C-4D09-B748-1D0478100FC0 */ (REFIID){0xC6,0x79,0xA3,0x5B,0x61,0x0C,0x4D,0x09,0xB7,0x48,0x1D,0x04,0x78,0x10,0x0F,0xC0}
#define IID_IDeckLinkAttributes                          /* ABC11843-D966-44CB-96E2-A1CB5D3135C4 */ (REFIID){0xAB,0xC1,0x18,0x43,0xD9,0x66,0x44,0xCB,0x96,0xE2,0xA1,0xCB,0x5D,0x31,0x35,0xC4}
#define IID_IDeckLinkKeyer                               /* 89AFCAF5-65F8-421E-98F7-96FE5F5BFBA3 */ (REFIID){0x89,0xAF,0xCA,0xF5,0x65,0xF8,0x42,0x1E,0x98,0xF7,0x96,0xFE,0x5F,0x5B,0xFB,0xA3}
#define IID_IDeckLinkVideoConversion                     /* 3BBCB8A2-DA2C-42D9-B5D8-88083644E99A */ (REFIID){0x3B,0xBC,0xB8,0xA2,0xDA,0x2C,0x42,0xD9,0xB5,0xD8,0x88,0x08,0x36,0x44,0xE9,0x9A}
#define IID_IDeckLinkDeckControlStatusCallback           /* E5F693C1-4283-4716-B18F-C1431521955B */ (REFIID){0xE5,0xF6,0x93,0xC1,0x42,0x83,0x47,0x16,0xB1,0x8F,0xC1,0x43,0x15,0x21,0x95,0x5B}
#define IID_IDeckLinkDeckControl                         /* A4D81043-0619-42B7-8ED6-602D29041DF7 */ (REFIID){0xA4,0xD8,0x10,0x43,0x06,0x19,0x42,0xB7,0x8E,0xD6,0x60,0x2D,0x29,0x04,0x1D,0xF7}


/* Enum BMDDisplayMode - Video display modes */

typedef uint32_t BMDDisplayMode;
enum _BMDDisplayMode {

    /* SD Modes */

    bmdModeNTSC                                        = /* 'ntsc' */ 0x6E747363,
    bmdModeNTSC2398                                    = /* 'nt23' */ 0x6E743233,	// 3:2 pulldown
    bmdModePAL                                         = /* 'pal ' */ 0x70616C20,

    /* HD 1080 Modes */

    bmdModeHD1080p2398                                 = /* '23ps' */ 0x32337073,
    bmdModeHD1080p24                                   = /* '24ps' */ 0x32347073,
    bmdModeHD1080p25                                   = /* 'Hp25' */ 0x48703235,
    bmdModeHD1080p2997                                 = /* 'Hp29' */ 0x48703239,
    bmdModeHD1080p30                                   = /* 'Hp30' */ 0x48703330,
    bmdModeHD1080i50                                   = /* 'Hi50' */ 0x48693530,
    bmdModeHD1080i5994                                 = /* 'Hi59' */ 0x48693539,
    bmdModeHD1080i6000                                 = /* 'Hi60' */ 0x48693630,	// N.B. This _really_ is 60.00 Hz.
    bmdModeHD1080p50                                   = /* 'Hp50' */ 0x48703530,
    bmdModeHD1080p5994                                 = /* 'Hp59' */ 0x48703539,
    bmdModeHD1080p6000                                 = /* 'Hp60' */ 0x48703630,	// N.B. This _really_ is 60.00 Hz.

    /* HD 720 Modes */

    bmdModeHD720p50                                    = /* 'hp50' */ 0x68703530,
    bmdModeHD720p5994                                  = /* 'hp59' */ 0x68703539,
    bmdModeHD720p60                                    = /* 'hp60' */ 0x68703630,

    /* 2k Modes */

    bmdMode2k2398                                      = /* '2k23' */ 0x326B3233,
    bmdMode2k24                                        = /* '2k24' */ 0x326B3234,
    bmdMode2k25                                        = /* '2k25' */ 0x326B3235
};


/* Enum BMDFieldDominance - Video field dominance */

typedef uint32_t BMDFieldDominance;
enum _BMDFieldDominance {
    bmdUnknownFieldDominance                           = 0,
    bmdLowerFieldFirst                                 = /* 'lowr' */ 0x6C6F7772,
    bmdUpperFieldFirst                                 = /* 'uppr' */ 0x75707072,
    bmdProgressiveFrame                                = /* 'prog' */ 0x70726F67,
    bmdProgressiveSegmentedFrame                       = /* 'psf ' */ 0x70736620
};


/* Enum BMDPixelFormat - Video pixel formats supported for output/input */

typedef uint32_t BMDPixelFormat;
enum _BMDPixelFormat {
    bmdFormat8BitYUV                                   = /* '2vuy' */ 0x32767579,
    bmdFormat10BitYUV                                  = /* 'v210' */ 0x76323130,
    bmdFormat8BitARGB                                  = 32,
    bmdFormat8BitBGRA                                  = /* 'BGRA' */ 0x42475241,
    bmdFormat10BitRGB                                  = /* 'r210' */ 0x72323130	// Big-endian RGB 10-bit per component with SMPTE video levels (64-960). Packed as 2:10:10:10
};


/* Enum BMDDisplayModeFlags - Flags to describe the characteristics of an IDeckLinkDisplayMode. */

typedef uint32_t BMDDisplayModeFlags;
enum _BMDDisplayModeFlags {
    bmdDisplayModeSupports3D                           = 1 << 0,
    bmdDisplayModeColorspaceRec601                     = 1 << 1,
    bmdDisplayModeColorspaceRec709                     = 1 << 2
};


/* Enum BMDVideoOutputFlags - Flags to control the output of ancillary data along with video. */

typedef uint32_t BMDVideoOutputFlags;
enum _BMDVideoOutputFlags {
    bmdVideoOutputFlagDefault                          = 0,
    bmdVideoOutputVANC                                 = 1 << 0,
    bmdVideoOutputVITC                                 = 1 << 1,
    bmdVideoOutputRP188                                = 1 << 2,
    bmdVideoOutputDualStream3D                         = 1 << 4
};


/* Enum BMDFrameFlags - Frame flags */

typedef uint32_t BMDFrameFlags;
enum _BMDFrameFlags {
    bmdFrameFlagDefault                                = 0,
    bmdFrameFlagFlipVertical                           = 1 << 0,

    /* Flags that are applicable only to instances of IDeckLinkVideoInputFrame */

    bmdFrameHasNoInputSource                           = 1 << 31
};


/* Enum BMDVideoInputFlags - Flags applicable to video input */

typedef uint32_t BMDVideoInputFlags;
enum _BMDVideoInputFlags {
    bmdVideoInputFlagDefault                           = 0,
    bmdVideoInputEnableFormatDetection                 = 1 << 0,
    bmdVideoInputDualStream3D                          = 1 << 1
};


/* Enum BMDVideoInputFormatChangedEvents - Bitmask passed to the VideoInputFormatChanged notification to identify the properties of the input signal that have changed */

typedef uint32_t BMDVideoInputFormatChangedEvents;
enum _BMDVideoInputFormatChangedEvents {
    bmdVideoInputDisplayModeChanged                    = 1 << 0,
    bmdVideoInputFieldDominanceChanged                 = 1 << 1,
    bmdVideoInputColorspaceChanged                     = 1 << 2
};


/* Enum BMDDetectedVideoInputFormatFlags - Flags passed to the VideoInputFormatChanged notification to describe the detected video input signal */

typedef uint32_t BMDDetectedVideoInputFormatFlags;
enum _BMDDetectedVideoInputFormatFlags {
    bmdDetectedVideoInputYCbCr422                      = 1 << 0,
    bmdDetectedVideoInputRGB444                        = 1 << 1
};


/* Enum BMDOutputFrameCompletionResult - Frame Completion Callback */

typedef uint32_t BMDOutputFrameCompletionResult;
enum _BMDOutputFrameCompletionResult {
    bmdOutputFrameCompleted,                          
    bmdOutputFrameDisplayedLate,                      
    bmdOutputFrameDropped,                            
    bmdOutputFrameFlushed                             
};


/* Enum BMDReferenceStatus - GenLock input status */

typedef uint32_t BMDReferenceStatus;
enum _BMDReferenceStatus {
    bmdReferenceNotSupportedByHardware                 = 1 << 0,
    bmdReferenceLocked                                 = 1 << 1
};


/* Enum BMDAudioSampleRate - Audio sample rates supported for output/input */

typedef uint32_t BMDAudioSampleRate;
enum _BMDAudioSampleRate {
    bmdAudioSampleRate48kHz                            = 48000
};


/* Enum BMDAudioSampleType - Audio sample sizes supported for output/input */

typedef uint32_t BMDAudioSampleType;
enum _BMDAudioSampleType {
    bmdAudioSampleType16bitInteger                     = 16,
    bmdAudioSampleType32bitInteger                     = 32
};


/* Enum BMDAudioOutputStreamType - Audio output stream type */

typedef uint32_t BMDAudioOutputStreamType;
enum _BMDAudioOutputStreamType {
    bmdAudioOutputStreamContinuous,                   
    bmdAudioOutputStreamContinuousDontResample,       
    bmdAudioOutputStreamTimestamped                   
};


/* Enum BMDDisplayModeSupport - Output mode supported flags */

typedef uint32_t BMDDisplayModeSupport;
enum _BMDDisplayModeSupport {
    bmdDisplayModeNotSupported                         = 0,
    bmdDisplayModeSupported,                          
    bmdDisplayModeSupportedWithConversion             
};


/* Enum BMDTimecodeFormat - Timecode formats for frame metadata */

typedef uint32_t BMDTimecodeFormat;
enum _BMDTimecodeFormat {
    bmdTimecodeRP188                                   = /* 'rp18' */ 0x72703138,
    bmdTimecodeVITC                                    = /* 'vitc' */ 0x76697463,
    bmdTimecodeSerial                                  = /* 'seri' */ 0x73657269
};


/* Enum BMDTimecodeFlags - Timecode flags */

typedef uint32_t BMDTimecodeFlags;
enum _BMDTimecodeFlags {
    bmdTimecodeFlagDefault                             = 0,
    bmdTimecodeIsDropFrame                             = 1 << 0
};


/* Enum BMDVideoConnection - Video connection types */

typedef uint32_t BMDVideoConnection;
enum _BMDVideoConnection {
    bmdVideoConnectionSDI                              = 1 << 0,
    bmdVideoConnectionHDMI                             = 1 << 1,
    bmdVideoConnectionOpticalSDI                       = 1 << 2,
    bmdVideoConnectionComponent                        = 1 << 3,
    bmdVideoConnectionComposite                        = 1 << 4,
    bmdVideoConnectionSVideo                           = 1 << 5
};


/* Enum BMDAnalogVideoFlags - Analog video display flags */

typedef uint32_t BMDAnalogVideoFlags;
enum _BMDAnalogVideoFlags {
    bmdAnalogVideoFlagCompositeSetup75                 = 1 << 0,
    bmdAnalogVideoFlagComponentBetacamLevels           = 1 << 1
};


/* Enum BMDAudioConnection - Audio connection types */

typedef uint32_t BMDAudioConnection;
enum _BMDAudioConnection {
    bmdAudioConnectionEmbedded                         = /* 'embd' */ 0x656D6264,
    bmdAudioConnectionAESEBU                           = /* 'aes ' */ 0x61657320,
    bmdAudioConnectionAnalog                           = /* 'anlg' */ 0x616E6C67
};


/* Enum BMDAudioOutputAnalogAESSwitch - Audio output Analog/AESEBU switch */

typedef uint32_t BMDAudioOutputAnalogAESSwitch;
enum _BMDAudioOutputAnalogAESSwitch {
    bmdAudioOutputSwitchAESEBU                         = /* 'aes ' */ 0x61657320,
    bmdAudioOutputSwitchAnalog                         = /* 'anlg' */ 0x616E6C67
};


/* Enum BMDVideoOutputConversionMode - Video/audio conversion mode */

typedef uint32_t BMDVideoOutputConversionMode;
enum _BMDVideoOutputConversionMode {
    bmdNoVideoOutputConversion                         = /* 'none' */ 0x6E6F6E65,
    bmdVideoOutputLetterboxDownconversion              = /* 'ltbx' */ 0x6C746278,
    bmdVideoOutputAnamorphicDownconversion             = /* 'amph' */ 0x616D7068,
    bmdVideoOutputHD720toHD1080Conversion              = /* '720c' */ 0x37323063,
    bmdVideoOutputHardwareLetterboxDownconversion      = /* 'HWlb' */ 0x48576C62,
    bmdVideoOutputHardwareAnamorphicDownconversion     = /* 'HWam' */ 0x4857616D,
    bmdVideoOutputHardwareCenterCutDownconversion      = /* 'HWcc' */ 0x48576363,
    bmdVideoOutputHardware720p1080pCrossconversion     = /* 'xcap' */ 0x78636170,
    bmdVideoOutputHardwareAnamorphic720pUpconversion   = /* 'ua7p' */ 0x75613770,
    bmdVideoOutputHardwareAnamorphic1080iUpconversion  = /* 'ua1i' */ 0x75613169,
    bmdVideoOutputHardwareAnamorphic149To720pUpconversion = /* 'u47p' */ 0x75343770,
    bmdVideoOutputHardwareAnamorphic149To1080iUpconversion = /* 'u41i' */ 0x75343169,
    bmdVideoOutputHardwarePillarbox720pUpconversion    = /* 'up7p' */ 0x75703770,
    bmdVideoOutputHardwarePillarbox1080iUpconversion   = /* 'up1i' */ 0x75703169
};


/* Enum BMDVideoInputConversionMode - Video input conversion mode */

typedef uint32_t BMDVideoInputConversionMode;
enum _BMDVideoInputConversionMode {
    bmdNoVideoInputConversion                          = /* 'none' */ 0x6E6F6E65,
    bmdVideoInputLetterboxDownconversionFromHD1080     = /* '10lb' */ 0x31306C62,
    bmdVideoInputAnamorphicDownconversionFromHD1080    = /* '10am' */ 0x3130616D,
    bmdVideoInputLetterboxDownconversionFromHD720      = /* '72lb' */ 0x37326C62,
    bmdVideoInputAnamorphicDownconversionFromHD720     = /* '72am' */ 0x3732616D,
    bmdVideoInputLetterboxUpconversion                 = /* 'lbup' */ 0x6C627570,
    bmdVideoInputAnamorphicUpconversion                = /* 'amup' */ 0x616D7570
};


/* Enum BMDVideo3DPackingFormat - Video 3D packing format */

typedef uint32_t BMDVideo3DPackingFormat;
enum _BMDVideo3DPackingFormat {
    bmdVideo3DPackingSidebySideHalf                    = /* 'sbsh' */ 0x73627368,
    bmdVideo3DPackingLinebyLine                        = /* 'lbyl' */ 0x6C62796C,
    bmdVideo3DPackingTopAndBottom                      = /* 'tabo' */ 0x7461626F,
    bmdVideo3DPackingLeftOnly                          = /* 'left' */ 0x6C656674,
    bmdVideo3DPackingRightOnly                         = /* 'righ' */ 0x72696768
};


/* Enum BMDDeckLinkConfigurationID - DeckLink Configuration ID */

typedef uint32_t BMDDeckLinkConfigurationID;
enum _BMDDeckLinkConfigurationID {

    /* Video Input/Output Flags */

    bmdDeckLinkConfigUse1080pNotPsF                    = /* 'fpro' */ 0x6670726F,

    /* Video Input/Output Integers */

    bmdDeckLinkConfigHDMI3DPackingFormat               = /* '3dpf' */ 0x33647066,

    /* Audio Input/Output Flags */

    bmdDeckLinkConfigAnalogAudioConsumerLevels         = /* 'aacl' */ 0x6161636C,

    /* Video output flags */

    bmdDeckLinkConfigFieldFlickerRemoval               = /* 'fdfr' */ 0x66646672,
    bmdDeckLinkConfigHD1080p24ToHD1080i5994Conversion  = /* 'to59' */ 0x746F3539,
    bmdDeckLinkConfig444SDIVideoOutput                 = /* '444o' */ 0x3434346F,
    bmdDeckLinkConfig3GBpsVideoOutput                  = /* '3gbs' */ 0x33676273,
    bmdDeckLinkConfigBlackVideoOutputDuringCapture     = /* 'bvoc' */ 0x62766F63,
    bmdDeckLinkConfigLowLatencyVideoOutput             = /* 'llvo' */ 0x6C6C766F,

    /* Video Output Integers */

    bmdDeckLinkConfigVideoOutputConnection             = /* 'vocn' */ 0x766F636E,
    bmdDeckLinkConfigVideoOutputConversionMode         = /* 'vocm' */ 0x766F636D,
    bmdDeckLinkConfigAnalogVideoOutputFlags            = /* 'avof' */ 0x61766F66,
    bmdDeckLinkConfigReferenceInputTimingOffset        = /* 'glot' */ 0x676C6F74,

    /* Video Input Integers */

    bmdDeckLinkConfigVideoInputConnection              = /* 'vicn' */ 0x7669636E,
    bmdDeckLinkConfigAnalogVideoInputFlags             = /* 'avif' */ 0x61766966,
    bmdDeckLinkConfigVideoInputConversionMode          = /* 'vicm' */ 0x7669636D,
    bmdDeckLinkConfig32PulldownSequenceInitialTimecodeFrame = /* 'pdif' */ 0x70646966,
    bmdDeckLinkConfigVANCSourceLine1Mapping            = /* 'vsl1' */ 0x76736C31,
    bmdDeckLinkConfigVANCSourceLine2Mapping            = /* 'vsl2' */ 0x76736C32,
    bmdDeckLinkConfigVANCSourceLine3Mapping            = /* 'vsl3' */ 0x76736C33,

    /* Audio Input Integers */

    bmdDeckLinkConfigAudioInputConnection              = /* 'aicn' */ 0x6169636E,

    /* Audio Input Floats */

    bmdDeckLinkConfigAnalogAudioInputScaleChannel1     = /* 'ais1' */ 0x61697331,
    bmdDeckLinkConfigAnalogAudioInputScaleChannel2     = /* 'ais2' */ 0x61697332,
    bmdDeckLinkConfigAnalogAudioInputScaleChannel3     = /* 'ais3' */ 0x61697333,
    bmdDeckLinkConfigAnalogAudioInputScaleChannel4     = /* 'ais4' */ 0x61697334,
    bmdDeckLinkConfigDigitalAudioInputScale            = /* 'dais' */ 0x64616973,

    /* Audio Output Integers */

    bmdDeckLinkConfigAudioOutputAESAnalogSwitch        = /* 'aoaa' */ 0x616F6161,

    /* Audio Output Floats */

    bmdDeckLinkConfigAnalogAudioOutputScaleChannel1    = /* 'aos1' */ 0x616F7331,
    bmdDeckLinkConfigAnalogAudioOutputScaleChannel2    = /* 'aos2' */ 0x616F7332,
    bmdDeckLinkConfigAnalogAudioOutputScaleChannel3    = /* 'aos3' */ 0x616F7333,
    bmdDeckLinkConfigAnalogAudioOutputScaleChannel4    = /* 'aos4' */ 0x616F7334,
    bmdDeckLinkConfigDigitalAudioOutputScale           = /* 'daos' */ 0x64616F73
};


/* Enum BMDDeckLinkAttributeID - DeckLink Attribute ID */

typedef uint32_t BMDDeckLinkAttributeID;
enum _BMDDeckLinkAttributeID {

    /* Flags */

    BMDDeckLinkSupportsInternalKeying                  = /* 'keyi' */ 0x6B657969,
    BMDDeckLinkSupportsExternalKeying                  = /* 'keye' */ 0x6B657965,
    BMDDeckLinkSupportsHDKeying                        = /* 'keyh' */ 0x6B657968,
    BMDDeckLinkSupportsInputFormatDetection            = /* 'infd' */ 0x696E6664,
    BMDDeckLinkHasReferenceInput                       = /* 'hrin' */ 0x6872696E,
    BMDDeckLinkHasSerialPort                           = /* 'hspt' */ 0x68737074,

    /* Integers */

    BMDDeckLinkMaximumAudioChannels                    = /* 'mach' */ 0x6D616368,
    BMDDeckLinkNumberOfSubDevices                      = /* 'nsbd' */ 0x6E736264,
    BMDDeckLinkSubDeviceIndex                          = /* 'subi' */ 0x73756269,
    BMDDeckLinkVideoOutputConnections                  = /* 'vocn' */ 0x766F636E,
    BMDDeckLinkVideoInputConnections                   = /* 'vicn' */ 0x7669636E,

    /* Strings */

    BMDDeckLinkSerialPortDeviceName                    = /* 'slpn' */ 0x736C706E
};


/* Enum BMDDeckLinkAPIInformationID - DeckLinkAPI information ID */

typedef uint32_t BMDDeckLinkAPIInformationID;
enum _BMDDeckLinkAPIInformationID {
    BMDDeckLinkAPIVersion                              = /* 'vers' */ 0x76657273
};


/* Enum BMDDeckControlMode - DeckControl mode */

typedef uint32_t BMDDeckControlMode;
enum _BMDDeckControlMode {
    bmdDeckControlNotOpened                            = /* 'ntop' */ 0x6E746F70,
    bmdDeckControlVTRControlMode                       = /* 'vtrc' */ 0x76747263,
    bmdDeckControlExportMode                           = /* 'expm' */ 0x6578706D,
    bmdDeckControlCaptureMode                          = /* 'capm' */ 0x6361706D
};


/* Enum BMDDeckControlEvent - DeckControl event */

typedef uint32_t BMDDeckControlEvent;
enum _BMDDeckControlEvent {
    bmdDeckControlAbortedEvent                         = /* 'abte' */ 0x61627465,	// This event is triggered when a capture or edit-to-tape operation is aborted.

    /* Export-To-Tape events */

    bmdDeckControlPrepareForExportEvent                = /* 'pfee' */ 0x70666565,	// This event is triggered a few frames before reaching the in-point. IDeckLinkInput::StartScheduledPlayback() should be called at this point.
    bmdDeckControlExportCompleteEvent                  = /* 'exce' */ 0x65786365,	// This event is triggered a few frames after reaching the out-point. At this point, it is safe to stop playback.

    /* Capture events */

    bmdDeckControlPrepareForCaptureEvent               = /* 'pfce' */ 0x70666365,	// This event is triggered a few frames before reaching the in-point. The serial timecode attached to IDeckLinkVideoInputFrames is now valid.
    bmdDeckControlCaptureCompleteEvent                 = /* 'ccev' */ 0x63636576	// This event is triggered a few frames after reaching the out-point.
};


/* Enum BMDDeckControlVTRControlState - VTR Control state */

typedef uint32_t BMDDeckControlVTRControlState;
enum _BMDDeckControlVTRControlState {
    bmdDeckControlNotInVTRControlMode                  = /* 'nvcm' */ 0x6E76636D,
    bmdDeckControlVTRControlPlaying                    = /* 'vtrp' */ 0x76747270,
    bmdDeckControlVTRControlRecording                  = /* 'vtrr' */ 0x76747272,
    bmdDeckControlVTRControlStill                      = /* 'vtra' */ 0x76747261,
    bmdDeckControlVTRControlSeeking                    = /* 'vtrs' */ 0x76747273,
    bmdDeckControlVTRControlStopped                    = /* 'vtro' */ 0x7674726F
};


/* Enum BMDDeckControlStatusFlags - Deck Control status flags */

typedef uint32_t BMDDeckControlStatusFlags;
enum _BMDDeckControlStatusFlags {
    bmdDeckControlStatusDeckConnected                  = 1 << 0,
    bmdDeckControlStatusRemoteMode                     = 1 << 1,
    bmdDeckControlStatusRecordInhibited                = 1 << 2,
    bmdDeckControlStatusCassetteOut                    = 1 << 3
};


/* Enum BMDDeckControlExportModeOpsFlags - Export mode flags */

typedef uint32_t BMDDeckControlExportModeOpsFlags;
enum _BMDDeckControlExportModeOpsFlags {
    bmdDeckControlExportModeInsertVideo                = 1 << 0,
    bmdDeckControlExportModeInsertAudio1               = 1 << 1,
    bmdDeckControlExportModeInsertAudio2               = 1 << 2,
    bmdDeckControlExportModeInsertAudio3               = 1 << 3,
    bmdDeckControlExportModeInsertAudio4               = 1 << 4,
    bmdDeckControlExportModeInsertAudio5               = 1 << 5,
    bmdDeckControlExportModeInsertAudio6               = 1 << 6,
    bmdDeckControlExportModeInsertAudio7               = 1 << 7,
    bmdDeckControlExportModeInsertAudio8               = 1 << 8,
    bmdDeckControlExportModeInsertAudio9               = 1 << 9,
    bmdDeckControlExportModeInsertAudio10              = 1 << 10,
    bmdDeckControlExportModeInsertAudio11              = 1 << 11,
    bmdDeckControlExportModeInsertAudio12              = 1 << 12,
    bmdDeckControlExportModeInsertTimeCode             = 1 << 13,
    bmdDeckControlExportModeInsertAssemble             = 1 << 14,
    bmdDeckControlExportModeInsertPreview              = 1 << 15,
    bmdDeckControlUseManualExport                      = 1 << 16
};


/* Enum BMDDeckControlError - Deck Control error */

typedef uint32_t BMDDeckControlError;
enum _BMDDeckControlError {
    bmdDeckControlNoError                              = /* 'noer' */ 0x6E6F6572,
    bmdDeckControlModeError                            = /* 'moer' */ 0x6D6F6572,
    bmdDeckControlMissedInPointError                   = /* 'mier' */ 0x6D696572,
    bmdDeckControlDeckTimeoutError                     = /* 'dter' */ 0x64746572,
    bmdDeckControlCommandFailedError                   = /* 'cfer' */ 0x63666572,
    bmdDeckControlDeviceAlreadyOpenedError             = /* 'dalo' */ 0x64616C6F,
    bmdDeckControlFailedToOpenDeviceError              = /* 'fder' */ 0x66646572,
    bmdDeckControlInLocalModeError                     = /* 'lmer' */ 0x6C6D6572,
    bmdDeckControlEndOfTapeError                       = /* 'eter' */ 0x65746572,
    bmdDeckControlUserAbortError                       = /* 'uaer' */ 0x75616572,
    bmdDeckControlNoTapeInDeckError                    = /* 'nter' */ 0x6E746572,
    bmdDeckControlNoVideoFromCardError                 = /* 'nvfc' */ 0x6E766663,
    bmdDeckControlNoCommunicationError                 = /* 'ncom' */ 0x6E636F6D,
    bmdDeckControlUnknownError                         = /* 'uner' */ 0x756E6572
};


/* Enum BMD3DPreviewFormat - Linked Frame preview format */

typedef uint32_t BMD3DPreviewFormat;
enum _BMD3DPreviewFormat {
    bmd3DPreviewFormatDefault                          = /* 'defa' */ 0x64656661,
    bmd3DPreviewFormatLeftOnly                         = /* 'left' */ 0x6C656674,
    bmd3DPreviewFormatRightOnly                        = /* 'righ' */ 0x72696768,
    bmd3DPreviewFormatSideBySide                       = /* 'side' */ 0x73696465,
    bmd3DPreviewFormatTopBottom                        = /* 'topb' */ 0x746F7062
};


#if defined(__cplusplus)

// Forward Declarations

class IDeckLinkVideoOutputCallback;
class IDeckLinkInputCallback;
class IDeckLinkMemoryAllocator;
class IDeckLinkAudioOutputCallback;
class IDeckLinkIterator;
class IDeckLinkAPIInformation;
class IDeckLinkDisplayModeIterator;
class IDeckLinkDisplayMode;
class IDeckLink;
class IDeckLinkOutput;
class IDeckLinkInput;
class IDeckLinkTimecode;
class IDeckLinkVideoFrame;
class IDeckLinkMutableVideoFrame;
class IDeckLinkVideoFrame3DExtensions;
class IDeckLinkVideoInputFrame;
class IDeckLinkVideoFrameAncillary;
class IDeckLinkAudioInputPacket;
class IDeckLinkScreenPreviewCallback;
class IDeckLinkGLScreenPreviewHelper;
class IDeckLinkConfiguration;
class IDeckLinkAttributes;
class IDeckLinkKeyer;
class IDeckLinkVideoConversion;
class IDeckLinkDeckControlStatusCallback;
class IDeckLinkDeckControl;


/* Interface IDeckLinkVideoOutputCallback - Frame completion callback. */

class IDeckLinkVideoOutputCallback : public IUnknown
{
public:
    virtual HRESULT ScheduledFrameCompleted (/* in */ IDeckLinkVideoFrame *completedFrame, /* in */ BMDOutputFrameCompletionResult result) = 0;
    virtual HRESULT ScheduledPlaybackHasStopped (void) = 0;

protected:
    virtual ~IDeckLinkVideoOutputCallback () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkInputCallback - Frame arrival callback. */

class IDeckLinkInputCallback : public IUnknown
{
public:
    virtual HRESULT VideoInputFormatChanged (/* in */ BMDVideoInputFormatChangedEvents notificationEvents, /* in */ IDeckLinkDisplayMode *newDisplayMode, /* in */ BMDDetectedVideoInputFormatFlags detectedSignalFlags) = 0;
    virtual HRESULT VideoInputFrameArrived (/* in */ IDeckLinkVideoInputFrame* videoFrame, /* in */ IDeckLinkAudioInputPacket* audioPacket) = 0;

protected:
    virtual ~IDeckLinkInputCallback () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkMemoryAllocator - Memory allocator for video frames. */

class IDeckLinkMemoryAllocator : public IUnknown
{
public:
    virtual HRESULT AllocateBuffer (/* in */ uint32_t bufferSize, /* out */ void **allocatedBuffer) = 0;
    virtual HRESULT ReleaseBuffer (/* in */ void *buffer) = 0;

    virtual HRESULT Commit (void) = 0;
    virtual HRESULT Decommit (void) = 0;
};


/* Interface IDeckLinkAudioOutputCallback - Optional callback to allow audio samples to be pulled as required. */

class IDeckLinkAudioOutputCallback : public IUnknown
{
public:
    virtual HRESULT RenderAudioSamples (/* in */ bool preroll) = 0;
};


/* Interface IDeckLinkIterator - enumerates installed DeckLink hardware */

class IDeckLinkIterator : public IUnknown
{
public:
    virtual HRESULT Next (/* out */ IDeckLink **deckLinkInstance) = 0;
};


/* Interface IDeckLinkAPIInformation - DeckLinkAPI attribute interface */

class IDeckLinkAPIInformation : public IUnknown
{
public:
    virtual HRESULT GetFlag (/* in */ BMDDeckLinkAPIInformationID cfgID, /* out */ bool *value) = 0;
    virtual HRESULT GetInt (/* in */ BMDDeckLinkAPIInformationID cfgID, /* out */ int64_t *value) = 0;
    virtual HRESULT GetFloat (/* in */ BMDDeckLinkAPIInformationID cfgID, /* out */ double *value) = 0;
    virtual HRESULT GetString (/* in */ BMDDeckLinkAPIInformationID cfgID, /* out */ const char **value) = 0;

protected:
    virtual ~IDeckLinkAPIInformation () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkDisplayModeIterator - enumerates over supported input/output display modes. */

class IDeckLinkDisplayModeIterator : public IUnknown
{
public:
    virtual HRESULT Next (/* out */ IDeckLinkDisplayMode **deckLinkDisplayMode) = 0;

protected:
    virtual ~IDeckLinkDisplayModeIterator () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkDisplayMode - represents a display mode */

class IDeckLinkDisplayMode : public IUnknown
{
public:
    virtual HRESULT GetName (/* out */ const char **name) = 0;
    virtual BMDDisplayMode GetDisplayMode (void) = 0;
    virtual long GetWidth (void) = 0;
    virtual long GetHeight (void) = 0;
    virtual HRESULT GetFrameRate (/* out */ BMDTimeValue *frameDuration, /* out */ BMDTimeScale *timeScale) = 0;
    virtual BMDFieldDominance GetFieldDominance (void) = 0;
    virtual BMDDisplayModeFlags GetFlags (void) = 0;

protected:
    virtual ~IDeckLinkDisplayMode () {}; // call Release method to drop reference count
};


/* Interface IDeckLink - represents a DeckLink device */

class IDeckLink : public IUnknown
{
public:
    virtual HRESULT GetModelName (/* out */ const char **modelName) = 0;
};


/* Interface IDeckLinkOutput - Created by QueryInterface from IDeckLink. */

class IDeckLinkOutput : public IUnknown
{
public:
    virtual HRESULT DoesSupportVideoMode (/* in */ BMDDisplayMode displayMode, /* in */ BMDPixelFormat pixelFormat, /* in */ BMDVideoOutputFlags flags, /* out */ BMDDisplayModeSupport *result, /* out */ IDeckLinkDisplayMode **resultDisplayMode) = 0;
    virtual HRESULT GetDisplayModeIterator (/* out */ IDeckLinkDisplayModeIterator **iterator) = 0;

    virtual HRESULT SetScreenPreviewCallback (/* in */ IDeckLinkScreenPreviewCallback *previewCallback) = 0;

    /* Video Output */

    virtual HRESULT EnableVideoOutput (/* in */ BMDDisplayMode displayMode, /* in */ BMDVideoOutputFlags flags) = 0;
    virtual HRESULT DisableVideoOutput (void) = 0;

    virtual HRESULT SetVideoOutputFrameMemoryAllocator (/* in */ IDeckLinkMemoryAllocator *theAllocator) = 0;
    virtual HRESULT CreateVideoFrame (/* in */ int32_t width, /* in */ int32_t height, /* in */ int32_t rowBytes, /* in */ BMDPixelFormat pixelFormat, /* in */ BMDFrameFlags flags, /* out */ IDeckLinkMutableVideoFrame **outFrame) = 0;
    virtual HRESULT CreateAncillaryData (/* in */ BMDPixelFormat pixelFormat, /* out */ IDeckLinkVideoFrameAncillary **outBuffer) = 0;

    virtual HRESULT DisplayVideoFrameSync (/* in */ IDeckLinkVideoFrame *theFrame) = 0;
    virtual HRESULT ScheduleVideoFrame (/* in */ IDeckLinkVideoFrame *theFrame, /* in */ BMDTimeValue displayTime, /* in */ BMDTimeValue displayDuration, /* in */ BMDTimeScale timeScale) = 0;
    virtual HRESULT SetScheduledFrameCompletionCallback (/* in */ IDeckLinkVideoOutputCallback *theCallback) = 0;
    virtual HRESULT GetBufferedVideoFrameCount (/* out */ uint32_t *bufferedFrameCount) = 0;

    /* Audio Output */

    virtual HRESULT EnableAudioOutput (/* in */ BMDAudioSampleRate sampleRate, /* in */ BMDAudioSampleType sampleType, /* in */ uint32_t channelCount, /* in */ BMDAudioOutputStreamType streamType) = 0;
    virtual HRESULT DisableAudioOutput (void) = 0;

    virtual HRESULT WriteAudioSamplesSync (/* in */ void *buffer, /* in */ uint32_t sampleFrameCount, /* out */ uint32_t *sampleFramesWritten) = 0;

    virtual HRESULT BeginAudioPreroll (void) = 0;
    virtual HRESULT EndAudioPreroll (void) = 0;
    virtual HRESULT ScheduleAudioSamples (/* in */ void *buffer, /* in */ uint32_t sampleFrameCount, /* in */ BMDTimeValue streamTime, /* in */ BMDTimeScale timeScale, /* out */ uint32_t *sampleFramesWritten) = 0;

    virtual HRESULT GetBufferedAudioSampleFrameCount (/* out */ uint32_t *bufferedSampleFrameCount) = 0;
    virtual HRESULT FlushBufferedAudioSamples (void) = 0;

    virtual HRESULT SetAudioCallback (/* in */ IDeckLinkAudioOutputCallback *theCallback) = 0;

    /* Output Control */

    virtual HRESULT StartScheduledPlayback (/* in */ BMDTimeValue playbackStartTime, /* in */ BMDTimeScale timeScale, /* in */ double playbackSpeed) = 0;
    virtual HRESULT StopScheduledPlayback (/* in */ BMDTimeValue stopPlaybackAtTime, /* out */ BMDTimeValue *actualStopTime, /* in */ BMDTimeScale timeScale) = 0;
    virtual HRESULT IsScheduledPlaybackRunning (/* out */ bool *active) = 0;
    virtual HRESULT GetScheduledStreamTime (/* in */ BMDTimeScale desiredTimeScale, /* out */ BMDTimeValue *streamTime, /* out */ double *playbackSpeed) = 0;
    virtual HRESULT GetReferenceStatus (/* out */ BMDReferenceStatus *referenceStatus) = 0;

    /* Hardware Timing */

    virtual HRESULT GetHardwareReferenceClock (/* in */ BMDTimeScale desiredTimeScale, /* out */ BMDTimeValue *hardwareTime, /* out */ BMDTimeValue *timeInFrame, /* out */ BMDTimeValue *ticksPerFrame) = 0;

protected:
    virtual ~IDeckLinkOutput () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkInput - Created by QueryInterface from IDeckLink. */

class IDeckLinkInput : public IUnknown
{
public:
    virtual HRESULT DoesSupportVideoMode (/* in */ BMDDisplayMode displayMode, /* in */ BMDPixelFormat pixelFormat, /* in */ BMDVideoInputFlags flags, /* out */ BMDDisplayModeSupport *result, /* out */ IDeckLinkDisplayMode **resultDisplayMode) = 0;
    virtual HRESULT GetDisplayModeIterator (/* out */ IDeckLinkDisplayModeIterator **iterator) = 0;

    virtual HRESULT SetScreenPreviewCallback (/* in */ IDeckLinkScreenPreviewCallback *previewCallback) = 0;

    /* Video Input */

    virtual HRESULT EnableVideoInput (/* in */ BMDDisplayMode displayMode, /* in */ BMDPixelFormat pixelFormat, /* in */ BMDVideoInputFlags flags) = 0;
    virtual HRESULT DisableVideoInput (void) = 0;
    virtual HRESULT GetAvailableVideoFrameCount (/* out */ uint32_t *availableFrameCount) = 0;

    /* Audio Input */

    virtual HRESULT EnableAudioInput (/* in */ BMDAudioSampleRate sampleRate, /* in */ BMDAudioSampleType sampleType, /* in */ uint32_t channelCount) = 0;
    virtual HRESULT DisableAudioInput (void) = 0;
    virtual HRESULT GetAvailableAudioSampleFrameCount (/* out */ uint32_t *availableSampleFrameCount) = 0;

    /* Input Control */

    virtual HRESULT StartStreams (void) = 0;
    virtual HRESULT StopStreams (void) = 0;
    virtual HRESULT PauseStreams (void) = 0;
    virtual HRESULT FlushStreams (void) = 0;
    virtual HRESULT SetCallback (/* in */ IDeckLinkInputCallback *theCallback) = 0;

    /* Hardware Timing */

    virtual HRESULT GetHardwareReferenceClock (/* in */ BMDTimeScale desiredTimeScale, /* out */ BMDTimeValue *hardwareTime, /* out */ BMDTimeValue *timeInFrame, /* out */ BMDTimeValue *ticksPerFrame) = 0;

protected:
    virtual ~IDeckLinkInput () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkTimecode - Used for video frame timecode representation. */

class IDeckLinkTimecode : public IUnknown
{
public:
    virtual BMDTimecodeBCD GetBCD (void) = 0;
    virtual HRESULT GetComponents (/* out */ uint8_t *hours, /* out */ uint8_t *minutes, /* out */ uint8_t *seconds, /* out */ uint8_t *frames) = 0;
    virtual HRESULT GetString (/* out */ const char **timecode) = 0;
    virtual BMDTimecodeFlags GetFlags (void) = 0;
    virtual HRESULT GetTimecodeUserBits (/* out */ BMDTimecodeUserBits *userBits) = 0;

protected:
    virtual ~IDeckLinkTimecode () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkVideoFrame - Interface to encapsulate a video frame; can be caller-implemented. */

class IDeckLinkVideoFrame : public IUnknown
{
public:
    virtual long GetWidth (void) = 0;
    virtual long GetHeight (void) = 0;
    virtual long GetRowBytes (void) = 0;
    virtual BMDPixelFormat GetPixelFormat (void) = 0;
    virtual BMDFrameFlags GetFlags (void) = 0;
    virtual HRESULT GetBytes (/* out */ void **buffer) = 0;

    virtual HRESULT GetTimecode (/* in */ BMDTimecodeFormat format, /* out */ IDeckLinkTimecode **timecode) = 0;
    virtual HRESULT GetAncillaryData (/* out */ IDeckLinkVideoFrameAncillary **ancillary) = 0;

protected:
    virtual ~IDeckLinkVideoFrame () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkMutableVideoFrame - Created by IDeckLinkOutput::CreateVideoFrame. */

class IDeckLinkMutableVideoFrame : public IDeckLinkVideoFrame
{
public:
    virtual HRESULT SetFlags (/* in */ BMDFrameFlags newFlags) = 0;

    virtual HRESULT SetTimecode (/* in */ BMDTimecodeFormat format, /* in */ IDeckLinkTimecode *timecode) = 0;
    virtual HRESULT SetTimecodeFromComponents (/* in */ BMDTimecodeFormat format, /* in */ uint8_t hours, /* in */ uint8_t minutes, /* in */ uint8_t seconds, /* in */ uint8_t frames, /* in */ BMDTimecodeFlags flags) = 0;
    virtual HRESULT SetAncillaryData (/* in */ IDeckLinkVideoFrameAncillary *ancillary) = 0;
    virtual HRESULT SetTimecodeUserBits (/* in */ BMDTimecodeFormat format, /* in */ BMDTimecodeUserBits userBits) = 0;

protected:
    virtual ~IDeckLinkMutableVideoFrame () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkVideoFrame3DExtensions - Optional interface implemented on IDeckLinkVideoFrame to support 3D frames */

class IDeckLinkVideoFrame3DExtensions : public IUnknown
{
public:
    virtual BMDVideo3DPackingFormat Get3DPackingFormat (void) = 0;
    virtual HRESULT GetFrameForRightEye (/* in */ IDeckLinkVideoFrame* *rightEyeFrame) = 0;

protected:
    virtual ~IDeckLinkVideoFrame3DExtensions () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkVideoInputFrame - Provided by the IDeckLinkVideoInput frame arrival callback. */

class IDeckLinkVideoInputFrame : public IDeckLinkVideoFrame
{
public:
    virtual HRESULT GetStreamTime (/* out */ BMDTimeValue *frameTime, /* out */ BMDTimeValue *frameDuration, /* in */ BMDTimeScale timeScale) = 0;
    virtual HRESULT GetHardwareReferenceTimestamp (/* in */ BMDTimeScale timeScale, /* out */ BMDTimeValue *frameTime, /* out */ BMDTimeValue *frameDuration) = 0;

protected:
    virtual ~IDeckLinkVideoInputFrame () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkVideoFrameAncillary - Obtained through QueryInterface() on an IDeckLinkVideoFrame object. */

class IDeckLinkVideoFrameAncillary : public IUnknown
{
public:

    virtual HRESULT GetBufferForVerticalBlankingLine (/* in */ uint32_t lineNumber, /* out */ void **buffer) = 0;
    virtual BMDPixelFormat GetPixelFormat (void) = 0;
    virtual BMDDisplayMode GetDisplayMode (void) = 0;

protected:
    virtual ~IDeckLinkVideoFrameAncillary () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkAudioInputPacket - Provided by the IDeckLinkInput callback. */

class IDeckLinkAudioInputPacket : public IUnknown
{
public:
    virtual long GetSampleFrameCount (void) = 0;
    virtual HRESULT GetBytes (/* out */ void **buffer) = 0;
    virtual HRESULT GetPacketTime (/* out */ BMDTimeValue *packetTime, /* in */ BMDTimeScale timeScale) = 0;

protected:
    virtual ~IDeckLinkAudioInputPacket () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkScreenPreviewCallback - Screen preview callback */

class IDeckLinkScreenPreviewCallback : public IUnknown
{
public:
    virtual HRESULT DrawFrame (/* in */ IDeckLinkVideoFrame *theFrame) = 0;

protected:
    virtual ~IDeckLinkScreenPreviewCallback () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkGLScreenPreviewHelper - Created with CoCreateInstance(). */

class IDeckLinkGLScreenPreviewHelper : public IUnknown
{
public:

    /* Methods must be called with OpenGL context set */

    virtual HRESULT InitializeGL (void) = 0;
    virtual HRESULT PaintGL (void) = 0;
    virtual HRESULT SetFrame (/* in */ IDeckLinkVideoFrame *theFrame) = 0;
    virtual HRESULT Set3DPreviewFormat (/* in */ BMD3DPreviewFormat previewFormat) = 0;

protected:
    virtual ~IDeckLinkGLScreenPreviewHelper () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkConfiguration - DeckLink Configuration interface */

class IDeckLinkConfiguration : public IUnknown
{
public:
    virtual HRESULT SetFlag (/* in */ BMDDeckLinkConfigurationID cfgID, /* in */ bool value) = 0;
    virtual HRESULT GetFlag (/* in */ BMDDeckLinkConfigurationID cfgID, /* out */ bool *value) = 0;
    virtual HRESULT SetInt (/* in */ BMDDeckLinkConfigurationID cfgID, /* in */ int64_t value) = 0;
    virtual HRESULT GetInt (/* in */ BMDDeckLinkConfigurationID cfgID, /* out */ int64_t *value) = 0;
    virtual HRESULT SetFloat (/* in */ BMDDeckLinkConfigurationID cfgID, /* in */ double value) = 0;
    virtual HRESULT GetFloat (/* in */ BMDDeckLinkConfigurationID cfgID, /* out */ double *value) = 0;
    virtual HRESULT SetString (/* in */ BMDDeckLinkConfigurationID cfgID, /* in */ const char *value) = 0;
    virtual HRESULT GetString (/* in */ BMDDeckLinkConfigurationID cfgID, /* out */ const char **value) = 0;
    virtual HRESULT WriteConfigurationToPreferences (void) = 0;

protected:
    virtual ~IDeckLinkConfiguration () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkAttributes - DeckLink Attribute interface */

class IDeckLinkAttributes : public IUnknown
{
public:
    virtual HRESULT GetFlag (/* in */ BMDDeckLinkAttributeID cfgID, /* out */ bool *value) = 0;
    virtual HRESULT GetInt (/* in */ BMDDeckLinkAttributeID cfgID, /* out */ int64_t *value) = 0;
    virtual HRESULT GetFloat (/* in */ BMDDeckLinkAttributeID cfgID, /* out */ double *value) = 0;
    virtual HRESULT GetString (/* in */ BMDDeckLinkAttributeID cfgID, /* out */ const char **value) = 0;

protected:
    virtual ~IDeckLinkAttributes () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkKeyer - DeckLink Keyer interface */

class IDeckLinkKeyer : public IUnknown
{
public:
    virtual HRESULT Enable (/* in */ bool isExternal) = 0;
    virtual HRESULT SetLevel (/* in */ uint8_t level) = 0;
    virtual HRESULT RampUp (/* in */ uint32_t numberOfFrames) = 0;
    virtual HRESULT RampDown (/* in */ uint32_t numberOfFrames) = 0;
    virtual HRESULT Disable (void) = 0;

protected:
    virtual ~IDeckLinkKeyer () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkVideoConversion - Created with CoCreateInstance(). */

class IDeckLinkVideoConversion : public IUnknown
{
public:
    virtual HRESULT ConvertFrame (/* in */ IDeckLinkVideoFrame* srcFrame, /* in */ IDeckLinkVideoFrame* dstFrame) = 0;

protected:
    virtual ~IDeckLinkVideoConversion () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkDeckControlStatusCallback - Deck control state change callback. */

class IDeckLinkDeckControlStatusCallback : public IUnknown
{
public:
    virtual HRESULT TimecodeUpdate (/* in */ BMDTimecodeBCD currentTimecode) = 0;
    virtual HRESULT VTRControlStateChanged (/* in */ BMDDeckControlVTRControlState newState, /* in */ BMDDeckControlError error) = 0;
    virtual HRESULT DeckControlEventReceived (/* in */ BMDDeckControlEvent event, /* in */ BMDDeckControlError error) = 0;
    virtual HRESULT DeckControlStatusChanged (/* in */ BMDDeckControlStatusFlags flags, /* in */ uint32_t mask) = 0;

protected:
    virtual ~IDeckLinkDeckControlStatusCallback () {}; // call Release method to drop reference count
};


/* Interface IDeckLinkDeckControl - Deck Control main interface */

class IDeckLinkDeckControl : public IUnknown
{
public:
    virtual HRESULT Open (/* in */ BMDTimeScale timeScale, /* in */ BMDTimeValue timeValue, /* in */ bool timecodeIsDropFrame, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Close (/* in */ bool standbyOn) = 0;
    virtual HRESULT GetCurrentState (/* out */ BMDDeckControlMode *mode, /* out */ BMDDeckControlVTRControlState *vtrControlState, /* out */ BMDDeckControlStatusFlags *flags) = 0;
    virtual HRESULT SetStandby (/* in */ bool standbyOn) = 0;
    virtual HRESULT Play (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Stop (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT TogglePlayStop (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Eject (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT GoToTimecode (/* in */ BMDTimecodeBCD timecode, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT FastForward (/* in */ bool viewTape, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Rewind (/* in */ bool viewTape, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT StepForward (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT StepBack (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Jog (/* in */ double rate, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Shuttle (/* in */ double rate, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT GetTimecodeString (/* out */ const char **currentTimeCode, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT GetTimecode (/* out */ IDeckLinkTimecode **currentTimecode, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT GetTimecodeBCD (/* out */ BMDTimecodeBCD *currentTimecode, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT SetPreroll (/* in */ uint32_t prerollSeconds) = 0;
    virtual HRESULT GetPreroll (/* out */ uint32_t *prerollSeconds) = 0;
    virtual HRESULT SetExportOffset (/* in */ int32_t exportOffsetFields) = 0;
    virtual HRESULT GetExportOffset (/* out */ int32_t *exportOffsetFields) = 0;
    virtual HRESULT GetManualExportOffset (/* out */ int32_t *deckManualExportOffsetFields) = 0;
    virtual HRESULT SetCaptureOffset (/* in */ int32_t captureOffsetFields) = 0;
    virtual HRESULT GetCaptureOffset (/* out */ int32_t *captureOffsetFields) = 0;
    virtual HRESULT StartExport (/* in */ BMDTimecodeBCD inTimecode, /* in */ BMDTimecodeBCD outTimecode, /* in */ BMDDeckControlExportModeOpsFlags exportModeOps, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT StartCapture (/* in */ bool useVITC, /* in */ BMDTimecodeBCD inTimecode, /* in */ BMDTimecodeBCD outTimecode, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT GetDeviceID (/* out */ uint16_t *deviceId, /* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT Abort (void) = 0;
    virtual HRESULT CrashRecordStart (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT CrashRecordStop (/* out */ BMDDeckControlError *error) = 0;
    virtual HRESULT SetCallback (/* in */ IDeckLinkDeckControlStatusCallback *callback) = 0;

protected:
    virtual ~IDeckLinkDeckControl () {}; // call Release method to drop reference count
};


/* Functions */

extern "C" {

    IDeckLinkIterator* CreateDeckLinkIteratorInstance (void);
    IDeckLinkGLScreenPreviewHelper* CreateOpenGLScreenPreviewHelper (void);
    IDeckLinkVideoConversion* CreateVideoConversionInstance (void);

};


#endif      // defined(__cplusplus)
#endif      // __DeckLink_API_h__
