import os
import subprocess

from twisted.internet.defer import inlineCallbacks

from juju.providers.local.network import Network, _SHELL_GET_LXC_DEFAULTS
from juju.lib.testing import TestCase
from juju.lib.mocker import ANY
from juju.errors import ProviderInteractionError

default_lxc_output = """\
LXC_BRIDGE=fakebridge0
LXC_ADDR=10.90.30.1
"""

class NetworkTestCase(TestCase):

    def _mock_default_lxc_exists(self):
        path_exists = self.mocker.mock()
        self.patch(os.path, 'exists', path_exists)
        path_exists('/etc/default/lxc')
        self.mocker.result(True)


    @inlineCallbacks
    def test_get_network_attributes(self):
        self._mock_default_lxc_exists()

        read_default_lxc = self.mocker.mock()
        self.patch(subprocess, 'check_output', read_default_lxc)
        read_default_lxc(['sh', '-c', _SHELL_GET_LXC_DEFAULTS])
        self.mocker.result(default_lxc_output)
        self.mocker.replay()

        network = Network()
        attributes = yield network.get_attributes()
        self.assertEqual(
            dict(ip="10.90.30.1", bridge="fakebridge0"), attributes)

    def _mock_status(self, expected_status):
        status_output_mock = self.mocker.mock()
        self.patch(subprocess, 'check_output', status_output_mock)
        status_output_mock(ANY)
        self.mocker.result(expected_status)

    def _mock_command(self, command):
        command_mock = self.mocker.mock()
        self.patch(subprocess, 'check_call', command_mock)
        command_mock([command, 'lxc-net'])
        return command_mock

    @inlineCallbacks
    def test_start_started_network(self):
        self._mock_status('lxc-net start/running')
        self.mocker.replay()

        network = Network()
        yield network.start()

    @inlineCallbacks
    def test_start_stoppped_network(self):
        self._mock_status('lxc-net stop/waiting')
        self._mock_command('start')
        self.mocker.replay()

        network = Network()
        yield network.start()

    @inlineCallbacks
    def test_stop_started_network(self):
        self._mock_status('lxc-net start/running')
        self._mock_command('stop')
        self.mocker.replay()

        network = Network()
        yield network.stop()

    @inlineCallbacks
    def test_stop_stopped_network(self):
        self._mock_status('lxc-net stop/waiting')
        self.mocker.replay()

        network = Network()
        yield network.stop()

    @inlineCallbacks
    def test_is_running(self):
        self._mock_status('lxc-net start/running')
        self.mocker.replay()

        network = Network()
        running = yield network.is_running()
        self.assertTrue(running)

    def test_bad_status(self):
        self._mock_status('nospaces')
        self.mocker.replay()

        network = Network()
        yield self.assertFailure(ProviderInteractionError, network.is_running())

    def _mock_subprocess_error(self, command, args):
        status_mock = self.mocker.mock()
        self.patch(subprocess, command, status_mock)
        status_mock(args)
        self.mocker.throw(subprocess.CalledProcessError(13, ' '.join(args)))
        self.mocker.replay()
        
    def test_upstart_status_error(self):
        self._mock_subprocess_error('check_output', ['status','lxc-net'])

        network = Network()
        yield self.assertFailure(ProviderInteractionError, network.is_running())

    @inlineCallbacks
    def test_upstart_start_stop_error(self):
        self._mock_status('lxc-net stop/waiting')
        self._mock_subprocess_error('check_call', ['start','lxc-net']) 

        log = self.capture_logging('juju.local-dev')
        network = Network()
        yield network.start()
        self.assertIn('Problem starting lxc-net', log.getvalue())

    @inlineCallbacks
    def test_get_network_attributes_sh_error(self):
        self._mock_default_lxc_exists()
        self._mock_subprocess_error('check_output',
            ['sh', '-c', _SHELL_GET_LXC_DEFAULTS])

        log = self.capture_logging('juju.local-dev')
        network = Network()
        yield network.get_attributes()
        self.assertIn('Problem reading values from /etc/default/lxc',
            log.getvalue())


