/*
 * Copyright (C) 2012 Hermann Meyer, Andreas Degert, Pete Shorthose, Steve Poskitt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_shimmizita.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "shimmizita.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace shimmizita {

class Gx_shimmizita_
{
private:
  // pointer to buffer
  float*      output;
  float*      input;
  float*      output1;
  float*      input1;
  // pointer to dsp class
  PluginLV2*  shimmizita;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_shimmizita_();
  ~Gx_shimmizita_();
};

// constructor
Gx_shimmizita_::Gx_shimmizita_() :
  output(NULL),
  input(NULL),
  shimmizita(shimmizita::plugin()) {};

// destructor
Gx_shimmizita_::~Gx_shimmizita_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (shimmizita->activate_plugin !=0)
    shimmizita->activate_plugin(false, shimmizita);
  // delete DSP class
  shimmizita->delete_instance(shimmizita);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_shimmizita_::init_dsp_(uint32_t rate)
{
  AVOIDDENORMALS(); // init the SSE denormal protection
  shimmizita->set_samplerate(rate, shimmizita); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_shimmizita_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    case EFFECTS_OUTPUT1:
      output1 = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT1:
      input1 = static_cast<float*>(data);
      break;
    default:
      break;
    }
}

void Gx_shimmizita_::activate_f()
{
  // allocate the internal DSP mem
  if (shimmizita->activate_plugin !=0)
    shimmizita->activate_plugin(true, shimmizita);
}

void Gx_shimmizita_::clean_up()
{
  // delete the internal DSP mem
  if (shimmizita->activate_plugin !=0)
    shimmizita->activate_plugin(false, shimmizita);
}

void Gx_shimmizita_::deactivate_f()
{
  // delete the internal DSP mem
  if (shimmizita->activate_plugin !=0)
    shimmizita->activate_plugin(false, shimmizita);
}

void Gx_shimmizita_::run_dsp_(uint32_t n_samples)
{
  if (n_samples< 1) return;
  shimmizita->stereo_audio(static_cast<int>(n_samples), input, input1, output, output1, shimmizita);
}

void Gx_shimmizita_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  shimmizita->connect_ports(port,  data, shimmizita);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_shimmizita_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_shimmizita_ *self = new Gx_shimmizita_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_shimmizita_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_shimmizita_*>(instance)->connect_all__ports(port, data);
}

void Gx_shimmizita_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_shimmizita_*>(instance)->activate_f();
}

void Gx_shimmizita_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_shimmizita_*>(instance)->run_dsp_(n_samples);
}

void Gx_shimmizita_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_shimmizita_*>(instance)->deactivate_f();
}

void Gx_shimmizita_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_shimmizita_* self = static_cast<Gx_shimmizita_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_shimmizita_::descriptor =
{
  GXPLUGIN_URI "#_shimmizita_",
  Gx_shimmizita_::instantiate,
  Gx_shimmizita_::connect_port,
  Gx_shimmizita_::activate,
  Gx_shimmizita_::run,
  Gx_shimmizita_::deactivate,
  Gx_shimmizita_::cleanup,
  NULL
};

} // end namespace shimmizita

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &shimmizita::Gx_shimmizita_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
