package regexp2

import (
	"fmt"
	"testing"
)

// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// https://github.com/mono/mono/blob/master/mcs/class/System/Test/System.Text.RegularExpressions/PerlTrials.cs
// ported from perl-5.6.1/t/op/re_tests

func TestMono_Basics(t *testing.T) {
	runRegexTrial(t, `abc`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc`, 0, "xbc", "Fail.")
	runRegexTrial(t, `abc`, 0, "axc", "Fail.")
	runRegexTrial(t, `abc`, 0, "abx", "Fail.")
	runRegexTrial(t, `abc`, 0, "xabcy", "Pass. Group[0]=(1,3)")
	runRegexTrial(t, `abc`, 0, "ababc", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `ab*c`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, 0, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab*bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `.{1}`, 0, "abbbbc", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `.{3,4}`, 0, "abbbbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab{0,}bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab+bc`, 0, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab+bc`, 0, "abc", "Fail.")
	runRegexTrial(t, `ab+bc`, 0, "abq", "Fail.")
	runRegexTrial(t, `ab{1,}bc`, 0, "abq", "Fail.")
	runRegexTrial(t, `ab+bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,}bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,3}bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{3,4}bc`, 0, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{4,5}bc`, 0, "abbbbc", "Fail.")
	runRegexTrial(t, `ab?bc`, 0, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab?bc`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}bc`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab?bc`, 0, "abbbbc", "Fail.")
	runRegexTrial(t, `ab?c`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}c`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, 0, "abcc", "Fail.")
	runRegexTrial(t, `^abc`, 0, "abcc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, 0, "aabc", "Fail.")
	runRegexTrial(t, `abc$`, 0, "aabc", "Pass. Group[0]=(1,3)")
	runRegexTrial(t, `abc$`, 0, "aabcd", "Fail.")
	runRegexTrial(t, `^`, 0, "abc", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `$`, 0, "abc", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `a.c`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.c`, 0, "axc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.*c`, 0, "axyzc", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `a.*c`, 0, "axyzd", "Fail.")
	runRegexTrial(t, `a[bc]d`, 0, "abc", "Fail.")
	runRegexTrial(t, `a[bc]d`, 0, "abd", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]e`, 0, "abd", "Fail.")
	runRegexTrial(t, `a[b-d]e`, 0, "ace", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]`, 0, "aac", "Pass. Group[0]=(1,2)")
	runRegexTrial(t, `a[-b]`, 0, "a-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-]`, 0, "a-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-a]`, 0, "-", "Error.")
	runRegexTrial(t, `a[]b`, 0, "-", "Error.")
	runRegexTrial(t, `a[`, 0, "-", "Error.")
	runRegexTrial(t, `a]`, 0, "a]", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[]]b`, 0, "a]b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, 0, "aed", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, 0, "abd", "Fail.")
	runRegexTrial(t, `a[^-b]c`, 0, "adc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^-b]c`, 0, "a-c", "Fail.")
	runRegexTrial(t, `a[^]b]c`, 0, "a]c", "Fail.")
	runRegexTrial(t, `a[^]b]c`, 0, "adc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `\ba\b`, 0, "a-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\ba\b`, 0, "-a", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\ba\b`, 0, "-a-", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\by\b`, 0, "xy", "Fail.")
	runRegexTrial(t, `\by\b`, 0, "yz", "Fail.")
	runRegexTrial(t, `\by\b`, 0, "xyz", "Fail.")
	runRegexTrial(t, `\Ba\B`, 0, "a-", "Fail.")
	runRegexTrial(t, `\Ba\B`, 0, "-a", "Fail.")
	runRegexTrial(t, `\Ba\B`, 0, "-a-", "Fail.")
	runRegexTrial(t, `\By\b`, 0, "xy", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\by\B`, 0, "yz", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\By\B`, 0, "xyz", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\w`, 0, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\w`, 0, "-", "Fail.")
	runRegexTrial(t, `\W`, 0, "a", "Fail.")
	runRegexTrial(t, `\W`, 0, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a\sb`, 0, "a b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a\sb`, 0, "a-b", "Fail.")
	runRegexTrial(t, `a\Sb`, 0, "a b", "Fail.")
	runRegexTrial(t, `a\Sb`, 0, "a-b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `\d`, 0, "1", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\d`, 0, "-", "Fail.")
	runRegexTrial(t, `\D`, 0, "1", "Fail.")
	runRegexTrial(t, `\D`, 0, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\w]`, 0, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\w]`, 0, "-", "Fail.")
	runRegexTrial(t, `[\W]`, 0, "a", "Fail.")
	runRegexTrial(t, `[\W]`, 0, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a[\s]b`, 0, "a b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[\s]b`, 0, "a-b", "Fail.")
	runRegexTrial(t, `a[\S]b`, 0, "a b", "Fail.")
	runRegexTrial(t, `a[\S]b`, 0, "a-b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `[\d]`, 0, "1", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\d]`, 0, "-", "Fail.")
	runRegexTrial(t, `[\D]`, 0, "1", "Fail.")
	runRegexTrial(t, `[\D]`, 0, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `ab|cd`, 0, "abc", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `ab|cd`, 0, "abcd", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `()ef`, 0, "def", "Pass. Group[0]=(1,2) Group[1]=(1,0)")
	runRegexTrial(t, `*a`, 0, "-", "Error.")
	runRegexTrial(t, `(*)b`, 0, "-", "Error.")
	runRegexTrial(t, `$b`, 0, "b", "Fail.")
	runRegexTrial(t, `a\`, 0, "-", "Error.")
	runRegexTrial(t, `a\(b`, 0, "a(b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a\(*b`, 0, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a\(*b`, 0, "a((b", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a\\b`, 0, "a\\b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc)`, 0, "-", "Error.")
	runRegexTrial(t, `(abc`, 0, "-", "Error.")
	runRegexTrial(t, `((a))`, 0, "abc", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)")
	runRegexTrial(t, `(a)b(c)`, 0, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)")
	runRegexTrial(t, `a+b+c`, 0, "aabbabc", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a{1,}b{1,}c`, 0, "aabbabc", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a**`, 0, "-", "Error.")
	runRegexTrial(t, `a.+?c`, 0, "abcabc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `(a+|b)*`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b){0,}`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b)+`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b){1,}`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b)?`, 0, "ab", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(a+|b){0,1}`, 0, "ab", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `)(`, 0, "-", "Error.")
	runRegexTrial(t, `[^ab]*`, 0, "cde", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc`, 0, "", "Fail.")
	runRegexTrial(t, `a*`, 0, "", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `([abc])*d`, 0, "abbbcd", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,1)(2,1)(3,1)(4,1)")
	runRegexTrial(t, `([abc])*bcd`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,1)")
	runRegexTrial(t, `a|b|c|d|e`, 0, "e", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(a|b|c|d|e)f`, 0, "ef", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `abcd*efg`, 0, "abcdefg", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `ab*`, 0, "xabyabbbz", "Pass. Group[0]=(1,2)")
	runRegexTrial(t, `ab*`, 0, "xayabbbz", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(ab|cd)e`, 0, "abcde", "Pass. Group[0]=(2,3) Group[1]=(2,2)")
	runRegexTrial(t, `[abhgefdc]ij`, 0, "hij", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^(ab|cd)e`, 0, "abcde", "Fail.")
	runRegexTrial(t, `(abc|)ef`, 0, "abcdef", "Pass. Group[0]=(4,2) Group[1]=(4,0)")
	runRegexTrial(t, `(a|b)c*d`, 0, "abcd", "Pass. Group[0]=(1,3) Group[1]=(1,1)")
	runRegexTrial(t, `(ab|ab*)bc`, 0, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)")
	runRegexTrial(t, `a([bc]*)c*`, 0, "abc", "Pass. Group[0]=(0,3) Group[1]=(1,2)")
	runRegexTrial(t, `a([bc]*)(c*d)`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)")
	runRegexTrial(t, `a([bc]+)(c*d)`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)")
	runRegexTrial(t, `a([bc]*)(c+d)`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)")
	runRegexTrial(t, `a[bcd]*dcdcde`, 0, "adcdcde", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `a[bcd]+dcdcde`, 0, "adcdcde", "Fail.")
	runRegexTrial(t, `(ab|a)b*c`, 0, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,2)")
	runRegexTrial(t, `((a)(b)c)(d)`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)")
	runRegexTrial(t, `[a-zA-Z_][a-zA-Z0-9_]*`, 0, "alpha", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `^a(bc+|b[eh])g|.h$`, 0, "abh", "Pass. Group[0]=(1,2) Group[1]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, 0, "effgz", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, 0, "ij", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, 0, "effg", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, 0, "bcdd", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, 0, "reffgz", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]=")
	runRegexTrial(t, `((((((((((a))))))))))`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `((((((((((a))))))))))\10`, 0, "aa", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `((((((((((a))))))))))!`, 0, "aa", "Fail.")
	runRegexTrial(t, `((((((((((a))))))))))!`, 0, "a!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `(((((((((a)))))))))`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)")
	runRegexTrial(t, `multiple words of text`, 0, "uh-uh", "Fail.")
	runRegexTrial(t, `multiple words`, 0, "multiple words, yeah", "Pass. Group[0]=(0,14)")
	runRegexTrial(t, `(.*)c(.*)`, 0, "abcde", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)")
	runRegexTrial(t, `\((.*), (.*)\)`, 0, "(a, b)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)")
	runRegexTrial(t, `[k]`, 0, "ab", "Fail.")
	runRegexTrial(t, `abcd`, 0, "abcd", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a(bc)d`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2)")
	runRegexTrial(t, `a[-]?c`, 0, "ac", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `(abc)\1`, 0, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(0,3)")
	runRegexTrial(t, `([a-c]*)\1`, 0, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(0,3)")
	runRegexTrial(t, `\1`, 0, "-", "Error.")
	runRegexTrial(t, `\2`, 0, "-", "Error.")
	runRegexTrial(t, `(a)|\1`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(a)|\1`, 0, "x", "Fail.")
	runRegexTrial(t, `(a)|\2`, 0, "-", "Error.")
	runRegexTrial(t, `(([a-c])b*?\2)*`, 0, "ababbbcbc", "Pass. Group[0]=(0,5) Group[1]=(0,3)(3,2) Group[2]=(0,1)(3,1)")
	runRegexTrial(t, `(([a-c])b*?\2){3}`, 0, "ababbbcbc", "Pass. Group[0]=(0,9) Group[1]=(0,3)(3,3)(6,3) Group[2]=(0,1)(3,1)(6,1)")
	runRegexTrial(t, `((\3|b)\2(a)x)+`, 0, "aaxabxbaxbbx", "Fail.")
	runRegexTrial(t, `((\3|b)\2(a)x)+`, 0, "aaaxabaxbaaxbbax", "Pass. Group[0]=(12,4) Group[1]=(12,4) Group[2]=(12,1) Group[3]=(14,1)")
	runRegexTrial(t, `((\3|b)\2(a)){2,}`, 0, "bbaababbabaaaaabbaaaabba", "Pass. Group[0]=(15,9) Group[1]=(15,3)(18,3)(21,3) Group[2]=(15,1)(18,1)(21,1) Group[3]=(17,1)(20,1)(23,1)")
	runRegexTrial(t, `abc`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc`, IgnoreCase, "XBC", "Fail.")
	runRegexTrial(t, `abc`, IgnoreCase, "AXC", "Fail.")
	runRegexTrial(t, `abc`, IgnoreCase, "ABX", "Fail.")
	runRegexTrial(t, `abc`, IgnoreCase, "XABCY", "Pass. Group[0]=(1,3)")
	runRegexTrial(t, `abc`, IgnoreCase, "ABABC", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `ab*c`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab*?bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{0,}?bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab+?bc`, IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab+bc`, IgnoreCase, "ABC", "Fail.")
	runRegexTrial(t, `ab+bc`, IgnoreCase, "ABQ", "Fail.")
	runRegexTrial(t, `ab{1,}bc`, IgnoreCase, "ABQ", "Fail.")
	runRegexTrial(t, `ab+bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,}?bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,3}?bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{3,4}?bc`, IgnoreCase, "ABBBBC", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{4,5}?bc`, IgnoreCase, "ABBBBC", "Fail.")
	runRegexTrial(t, `ab??bc`, IgnoreCase, "ABBC", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab??bc`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}?bc`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab??bc`, IgnoreCase, "ABBBBC", "Fail.")
	runRegexTrial(t, `ab??c`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}?c`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, IgnoreCase, "ABCC", "Fail.")
	runRegexTrial(t, `^abc`, IgnoreCase, "ABCC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, IgnoreCase, "AABC", "Fail.")
	runRegexTrial(t, `abc$`, IgnoreCase, "AABC", "Pass. Group[0]=(1,3)")
	runRegexTrial(t, `^`, IgnoreCase, "ABC", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `$`, IgnoreCase, "ABC", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `a.c`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.c`, IgnoreCase, "AXC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.*?c`, IgnoreCase, "AXYZC", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `a.*c`, IgnoreCase, "AXYZD", "Fail.")
	runRegexTrial(t, `a[bc]d`, IgnoreCase, "ABC", "Fail.")
	runRegexTrial(t, `a[bc]d`, IgnoreCase, "ABD", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]e`, IgnoreCase, "ABD", "Fail.")
	runRegexTrial(t, `a[b-d]e`, IgnoreCase, "ACE", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]`, IgnoreCase, "AAC", "Pass. Group[0]=(1,2)")
	runRegexTrial(t, `a[-b]`, IgnoreCase, "A-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-]`, IgnoreCase, "A-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-a]`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `a[]b`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `a[`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `a]`, IgnoreCase, "A]", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[]]b`, IgnoreCase, "A]B", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, IgnoreCase, "AED", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, IgnoreCase, "ABD", "Fail.")
	runRegexTrial(t, `a[^-b]c`, IgnoreCase, "ADC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^-b]c`, IgnoreCase, "A-C", "Fail.")
	runRegexTrial(t, `a[^]b]c`, IgnoreCase, "A]C", "Fail.")
	runRegexTrial(t, `a[^]b]c`, IgnoreCase, "ADC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab|cd`, IgnoreCase, "ABC", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `ab|cd`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `()ef`, IgnoreCase, "DEF", "Pass. Group[0]=(1,2) Group[1]=(1,0)")
	runRegexTrial(t, `*a`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `(*)b`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `$b`, IgnoreCase, "B", "Fail.")
	runRegexTrial(t, `a\`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `a\(b`, IgnoreCase, "A(B", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a\(*b`, IgnoreCase, "AB", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a\(*b`, IgnoreCase, "A((B", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a\\b`, IgnoreCase, "A\\B", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc)`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `(abc`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `((a))`, IgnoreCase, "ABC", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)")
	runRegexTrial(t, `(a)b(c)`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)")
	runRegexTrial(t, `a+b+c`, IgnoreCase, "AABBABC", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a{1,}b{1,}c`, IgnoreCase, "AABBABC", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a**`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `a.+?c`, IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.*?c`, IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.{0,5}?c`, IgnoreCase, "ABCABC", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `(a+|b)*`, IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b){0,}`, IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b)+`, IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b){1,}`, IgnoreCase, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)(1,1)")
	runRegexTrial(t, `(a+|b)?`, IgnoreCase, "AB", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(a+|b){0,1}`, IgnoreCase, "AB", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(a+|b){0,1}?`, IgnoreCase, "AB", "Pass. Group[0]=(0,0) Group[1]=")
	runRegexTrial(t, `)(`, IgnoreCase, "-", "Error.")
	runRegexTrial(t, `[^ab]*`, IgnoreCase, "CDE", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc`, IgnoreCase, "", "Fail.")
	runRegexTrial(t, `a*`, IgnoreCase, "", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `([abc])*d`, IgnoreCase, "ABBBCD", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,1)(2,1)(3,1)(4,1)")
	runRegexTrial(t, `([abc])*bcd`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(0,1)")
	runRegexTrial(t, `a|b|c|d|e`, IgnoreCase, "E", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(a|b|c|d|e)f`, IgnoreCase, "EF", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `abcd*efg`, IgnoreCase, "ABCDEFG", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `ab*`, IgnoreCase, "XABYABBBZ", "Pass. Group[0]=(1,2)")
	runRegexTrial(t, `ab*`, IgnoreCase, "XAYABBBZ", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(ab|cd)e`, IgnoreCase, "ABCDE", "Pass. Group[0]=(2,3) Group[1]=(2,2)")
	runRegexTrial(t, `[abhgefdc]ij`, IgnoreCase, "HIJ", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^(ab|cd)e`, IgnoreCase, "ABCDE", "Fail.")
	runRegexTrial(t, `(abc|)ef`, IgnoreCase, "ABCDEF", "Pass. Group[0]=(4,2) Group[1]=(4,0)")
	runRegexTrial(t, `(a|b)c*d`, IgnoreCase, "ABCD", "Pass. Group[0]=(1,3) Group[1]=(1,1)")
	runRegexTrial(t, `(ab|ab*)bc`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,1)")
	runRegexTrial(t, `a([bc]*)c*`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(1,2)")
	runRegexTrial(t, `a([bc]*)(c*d)`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)")
	runRegexTrial(t, `a([bc]+)(c*d)`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2) Group[2]=(3,1)")
	runRegexTrial(t, `a([bc]*)(c+d)`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)")
	runRegexTrial(t, `a[bcd]*dcdcde`, IgnoreCase, "ADCDCDE", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `a[bcd]+dcdcde`, IgnoreCase, "ADCDCDE", "Fail.")
	runRegexTrial(t, `(ab|a)b*c`, IgnoreCase, "ABC", "Pass. Group[0]=(0,3) Group[1]=(0,2)")
	runRegexTrial(t, `((a)(b)c)(d)`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)")
	runRegexTrial(t, `[a-zA-Z_][a-zA-Z0-9_]*`, IgnoreCase, "ALPHA", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `^a(bc+|b[eh])g|.h$`, IgnoreCase, "ABH", "Pass. Group[0]=(1,2) Group[1]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, IgnoreCase, "EFFGZ", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, IgnoreCase, "IJ", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, IgnoreCase, "EFFG", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, IgnoreCase, "BCDD", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, IgnoreCase, "REFFGZ", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]=")
	runRegexTrial(t, `((((((((((a))))))))))`, IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `((((((((((a))))))))))\10`, IgnoreCase, "AA", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `((((((((((a))))))))))!`, IgnoreCase, "AA", "Fail.")
	runRegexTrial(t, `((((((((((a))))))))))!`, IgnoreCase, "A!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `(((((((((a)))))))))`, IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)")
	runRegexTrial(t, `(?:(?:(?:(?:(?:(?:(?:(?:(?:(a))))))))))`, IgnoreCase, "A", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?:(?:(?:(?:(?:(?:(?:(?:(a|b|c))))))))))`, IgnoreCase, "C", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `multiple words of text`, IgnoreCase, "UH-UH", "Fail.")
	runRegexTrial(t, `multiple words`, IgnoreCase, "MULTIPLE WORDS, YEAH", "Pass. Group[0]=(0,14)")
	runRegexTrial(t, `(.*)c(.*)`, IgnoreCase, "ABCDE", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)")
	runRegexTrial(t, `\((.*), (.*)\)`, IgnoreCase, "(A, B)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)")
	runRegexTrial(t, `[k]`, IgnoreCase, "AB", "Fail.")
	runRegexTrial(t, `abcd`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a(bc)d`, IgnoreCase, "ABCD", "Pass. Group[0]=(0,4) Group[1]=(1,2)")
	runRegexTrial(t, `a[-]?c`, IgnoreCase, "AC", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `(abc)\1`, IgnoreCase, "ABCABC", "Pass. Group[0]=(0,6) Group[1]=(0,3)")
	runRegexTrial(t, `([a-c]*)\1`, IgnoreCase, "ABCABC", "Pass. Group[0]=(0,6) Group[1]=(0,3)")
	runRegexTrial(t, `a(?!b).`, 0, "abad", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `a(?=d).`, 0, "abad", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `a(?=c|d).`, 0, "abad", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `a(?:b|c|d)(.)`, 0, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)")
	runRegexTrial(t, `a(?:b|c|d)*(.)`, 0, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)")
	runRegexTrial(t, `a(?:b|c|d)+?(.)`, 0, "ace", "Pass. Group[0]=(0,3) Group[1]=(2,1)")
	runRegexTrial(t, `a(?:b|c|d)+?(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,3) Group[1]=(2,1)")
	runRegexTrial(t, `a(?:b|c|d)+(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)")
	runRegexTrial(t, `a(?:b|c|d){2}(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,4) Group[1]=(3,1)")
	runRegexTrial(t, `a(?:b|c|d){4,5}(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)")
	runRegexTrial(t, `a(?:b|c|d){4,5}?(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,6) Group[1]=(5,1)")
	runRegexTrial(t, `((foo)|(bar))*`, 0, "foobar", "Pass. Group[0]=(0,6) Group[1]=(0,3)(3,3) Group[2]=(0,3) Group[3]=(3,3)")
	runRegexTrial(t, `:(?:`, 0, "-", "Error.")
	runRegexTrial(t, `a(?:b|c|d){6,7}(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)")
	runRegexTrial(t, `a(?:b|c|d){6,7}?(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)")
	runRegexTrial(t, `a(?:b|c|d){5,6}(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)")
	runRegexTrial(t, `a(?:b|c|d){5,6}?(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)")
	runRegexTrial(t, `a(?:b|c|d){5,7}(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,8) Group[1]=(7,1)")
	runRegexTrial(t, `a(?:b|c|d){5,7}?(.)`, 0, "acdbcdbe", "Pass. Group[0]=(0,7) Group[1]=(6,1)")
	runRegexTrial(t, `a(?:b|(c|e){1,2}?|d)+?(.)`, 0, "ace", "Pass. Group[0]=(0,3) Group[1]=(1,1) Group[2]=(2,1)")
	runRegexTrial(t, `^(.+)?B`, 0, "AB", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `^([^a-z])|(\^)$`, 0, ".", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=")
	runRegexTrial(t, `^[<>]&`, 0, "<&OUT", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `^(a\1?){4}$`, 0, "aaaaaaaaaa", "Pass. Group[0]=(0,10) Group[1]=(0,1)(1,2)(3,3)(6,4)")
	runRegexTrial(t, `^(a\1?){4}$`, 0, "aaaaaaaaa", "Fail.")
	runRegexTrial(t, `^(a\1?){4}$`, 0, "aaaaaaaaaaa", "Fail.")
	runRegexTrial(t, `^(a(?(1)\1)){4}$`, 0, "aaaaaaaaaa", "Pass. Group[0]=(0,10) Group[1]=(0,1)(1,2)(3,3)(6,4)")
	runRegexTrial(t, `^(a(?(1)\1)){4}$`, 0, "aaaaaaaaa", "Fail.")
	runRegexTrial(t, `^(a(?(1)\1)){4}$`, 0, "aaaaaaaaaaa", "Fail.")
	runRegexTrial(t, `((a{4})+)`, 0, "aaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4)")
	runRegexTrial(t, `(((aa){2})+)`, 0, "aaaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4) Group[3]=(0,2)(2,2)(4,2)(6,2)")
	runRegexTrial(t, `(((a{2}){2})+)`, 0, "aaaaaaaaaa", "Pass. Group[0]=(0,8) Group[1]=(0,8) Group[2]=(0,4)(4,4) Group[3]=(0,2)(2,2)(4,2)(6,2)")
	runRegexTrial(t, `(?:(f)(o)(o)|(b)(a)(r))*`, 0, "foobar", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(1,1) Group[3]=(2,1) Group[4]=(3,1) Group[5]=(4,1) Group[6]=(5,1)")
	runRegexTrial(t, `(?<=a)b`, 0, "ab", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(?<=a)b`, 0, "cb", "Fail.")
	runRegexTrial(t, `(?<=a)b`, 0, "b", "Fail.")
	runRegexTrial(t, `(?<!c)b`, 0, "ab", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(?<!c)b`, 0, "cb", "Fail.")
	runRegexTrial(t, `(?<!c)b`, 0, "b", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(?<!c)b`, 0, "b", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(?<%)b`, 0, "-", "Error.")
	runRegexTrial(t, `(?:..)*a`, 0, "aba", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `(?:..)*?a`, 0, "aba", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `^(?:b|a(?=(.)))*\1`, 0, "abc", "Pass. Group[0]=(0,2) Group[1]=(1,1)")
	runRegexTrial(t, `^(){3,5}`, 0, "abc", "Pass. Group[0]=(0,0) Group[1]=(0,0)(0,0)(0,0)")
	runRegexTrial(t, `^(a+)*ax`, 0, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1)")
	runRegexTrial(t, `^((a|b)+)*ax`, 0, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(0,1)")
	runRegexTrial(t, `^((a|bc)+)*ax`, 0, "aax", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(0,1)")
	runRegexTrial(t, `(a|x)*ab`, 0, "cab", "Pass. Group[0]=(1,2) Group[1]=")
	runRegexTrial(t, `(a)*ab`, 0, "cab", "Pass. Group[0]=(1,2) Group[1]=")
	runRegexTrial(t, `(?:(?i)a)b`, 0, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?i)a)b`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?i)a)b`, 0, "Ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?i)a)b`, 0, "Ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?i)a)b`, 0, "aB", "Fail.")
	runRegexTrial(t, `((?i)a)b`, 0, "aB", "Fail.")
	runRegexTrial(t, `(?i:a)b`, 0, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?i:a))b`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?i:a)b`, 0, "Ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?i:a))b`, 0, "Ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?i:a)b`, 0, "aB", "Fail.")
	runRegexTrial(t, `((?i:a))b`, 0, "aB", "Fail.")
	runRegexTrial(t, `(?:(?-i)a)b`, IgnoreCase, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i)a)b`, IgnoreCase, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?-i)a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i)a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?-i)a)b`, IgnoreCase, "Ab", "Fail.")
	runRegexTrial(t, `((?-i)a)b`, IgnoreCase, "Ab", "Fail.")
	runRegexTrial(t, `(?:(?-i)a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i)a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?:(?-i)a)b`, IgnoreCase, "AB", "Fail.")
	runRegexTrial(t, `((?-i)a)b`, IgnoreCase, "AB", "Fail.")
	runRegexTrial(t, `(?-i:a)b`, IgnoreCase, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i:a))b`, IgnoreCase, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?-i:a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i:a))b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?-i:a)b`, IgnoreCase, "Ab", "Fail.")
	runRegexTrial(t, `((?-i:a))b`, IgnoreCase, "Ab", "Fail.")
	runRegexTrial(t, `(?-i:a)b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `((?-i:a))b`, IgnoreCase, "aB", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `(?-i:a)b`, IgnoreCase, "AB", "Fail.")
	runRegexTrial(t, `((?-i:a))b`, IgnoreCase, "AB", "Fail.")
	runRegexTrial(t, `((?-i:a.))b`, IgnoreCase, "a\nB", "Fail.")
	runRegexTrial(t, `((?s-i:a.))b`, IgnoreCase, "a\nB", "Pass. Group[0]=(0,3) Group[1]=(0,2)")
	runRegexTrial(t, `((?s-i:a.))b`, IgnoreCase, "B\nB", "Fail.")
	runRegexTrial(t, `(?:c|d)(?:)(?:a(?:)(?:b)(?:b(?:))(?:b(?:)(?:b)))`, 0, "cabbbb", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `(?:c|d)(?:)(?:aaaaaaaa(?:)(?:bbbbbbbb)(?:bbbbbbbb(?:))(?:bbbbbbbb(?:)(?:bbbbbbbb)))`, 0, "caaaaaaaabbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb", "Pass. Group[0]=(0,41)")
	runRegexTrial(t, `(ab)\d\1`, IgnoreCase, "Ab4ab", "Pass. Group[0]=(0,5) Group[1]=(0,2)")
	runRegexTrial(t, `(ab)\d\1`, IgnoreCase, "ab4Ab", "Pass. Group[0]=(0,5) Group[1]=(0,2)")
	runRegexTrial(t, `foo\w*\d{4}baz`, 0, "foobar1234baz", "Pass. Group[0]=(0,13)")
	runRegexTrial(t, `x(~~)*(?:(?:F)?)?`, 0, "x~~", "Pass. Group[0]=(0,3) Group[1]=(1,2)")
	runRegexTrial(t, `^a(?#xxx){3}c`, 0, "aaac", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `(?<![cd])b`, 0, "dbcb", "Fail.")
	runRegexTrial(t, `(?<![cd])[ab]`, 0, "dbaacb", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `(?<!(c|d))b`, 0, "dbcb", "Fail.")
	runRegexTrial(t, `(?<!(c|d))[ab]`, 0, "dbaacb", "Pass. Group[0]=(2,1) Group[1]=")
	runRegexTrial(t, `(?<!cd)[ab]`, 0, "cdaccb", "Pass. Group[0]=(5,1)")
	runRegexTrial(t, `^(?:a?b?)*$`, 0, "a--", "Fail.")
	runRegexTrial(t, `((?s)^a(.))((?m)^b$)`, 0, "a\nb\nc\n", "Pass. Group[0]=(0,3) Group[1]=(0,2) Group[2]=(1,1) Group[3]=(2,1)")
	runRegexTrial(t, `((?m)^b$)`, 0, "a\nb\nc\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)")
	runRegexTrial(t, `(?m)^b`, 0, "a\nb\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `(?m)^(b)`, 0, "a\nb\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)")
	runRegexTrial(t, `((?m)^b)`, 0, "a\nb\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)")
	runRegexTrial(t, `\n((?m)^b)`, 0, "a\nb\n", "Pass. Group[0]=(1,2) Group[1]=(2,1)")
	runRegexTrial(t, `((?s).)c(?!.)`, 0, "a\nb\nc\n", "Pass. Group[0]=(3,2) Group[1]=(3,1)")
	runRegexTrial(t, `((?s).)c(?!.)`, 0, "a\nb\nc\n", "Pass. Group[0]=(3,2) Group[1]=(3,1)")
	runRegexTrial(t, `((?s)b.)c(?!.)`, 0, "a\nb\nc\n", "Pass. Group[0]=(2,3) Group[1]=(2,2)")
	runRegexTrial(t, `((?s)b.)c(?!.)`, 0, "a\nb\nc\n", "Pass. Group[0]=(2,3) Group[1]=(2,2)")
	runRegexTrial(t, `^b`, 0, "a\nb\nc\n", "Fail.")
	runRegexTrial(t, `()^b`, 0, "a\nb\nc\n", "Fail.")
	runRegexTrial(t, `((?m)^b)`, 0, "a\nb\nc\n", "Pass. Group[0]=(2,1) Group[1]=(2,1)")
	runRegexTrial(t, `(x)?(?(1)a|b)`, 0, "a", "Fail.")
	runRegexTrial(t, `(x)?(?(1)b|a)`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=")
	runRegexTrial(t, `()?(?(1)b|a)`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=")
	runRegexTrial(t, `()(?(1)b|a)`, 0, "a", "Fail.")
	runRegexTrial(t, `()?(?(1)a|b)`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=(0,0)")
	runRegexTrial(t, `^(\()?blah(?(1)(\)))$`, 0, "(blah)", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(5,1)")
	runRegexTrial(t, `^(\()?blah(?(1)(\)))$`, 0, "blah", "Pass. Group[0]=(0,4) Group[1]= Group[2]=")
	runRegexTrial(t, `^(\()?blah(?(1)(\)))$`, 0, "blah)", "Fail.")
	runRegexTrial(t, `^(\()?blah(?(1)(\)))$`, 0, "(blah", "Fail.")
	runRegexTrial(t, `^(\(+)?blah(?(1)(\)))$`, 0, "(blah)", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(5,1)")
	runRegexTrial(t, `^(\(+)?blah(?(1)(\)))$`, 0, "blah", "Pass. Group[0]=(0,4) Group[1]= Group[2]=")
	runRegexTrial(t, `^(\(+)?blah(?(1)(\)))$`, 0, "blah)", "Fail.")
	runRegexTrial(t, `^(\(+)?blah(?(1)(\)))$`, 0, "(blah", "Fail.")
	runRegexTrial(t, `(?(1)a|b|c)`, 0, "a", "Error.")
	runRegexTrial(t, `(?(?!a)a|b)`, 0, "a", "Fail.")
	runRegexTrial(t, `(?(?!a)b|a)`, 0, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(?(?=a)b|a)`, 0, "a", "Fail.")
	runRegexTrial(t, `(?(?=a)a|b)`, 0, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(?=(a+?))(\1ab)`, 0, "aaab", "Pass. Group[0]=(1,3) Group[1]=(1,1) Group[2]=(1,3)")
	runRegexTrial(t, `^(?=(a+?))\1ab`, 0, "aaab", "Fail.")
	runRegexTrial(t, `(\w+:)+`, 0, "one:", "Pass. Group[0]=(0,4) Group[1]=(0,4)")
	runRegexTrial(t, `$(?<=^(a))`, 0, "a", "Pass. Group[0]=(1,0) Group[1]=(0,1)")
	runRegexTrial(t, `(?=(a+?))(\1ab)`, 0, "aaab", "Pass. Group[0]=(1,3) Group[1]=(1,1) Group[2]=(1,3)")
	runRegexTrial(t, `^(?=(a+?))\1ab`, 0, "aaab", "Fail.")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "abcd:", "Fail.")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]= Group[2]=(0,4)")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "xy:z:::abcd", "Pass. Group[0]=(0,11) Group[1]=(0,7) Group[2]=(7,4)")
	runRegexTrial(t, `^[^bcd]*(c+)`, 0, "aexycd", "Pass. Group[0]=(0,5) Group[1]=(4,1)")
	runRegexTrial(t, `(a*)b+`, 0, "caab", "Pass. Group[0]=(1,3) Group[1]=(1,2)")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "abcd:", "Fail.")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "abcd", "Pass. Group[0]=(0,4) Group[1]= Group[2]=(0,4)")
	runRegexTrial(t, `([\w:]+::)?(\w+)$`, 0, "xy:z:::abcd", "Pass. Group[0]=(0,11) Group[1]=(0,7) Group[2]=(7,4)")
	runRegexTrial(t, `^[^bcd]*(c+)`, 0, "aexycd", "Pass. Group[0]=(0,5) Group[1]=(4,1)")
	runRegexTrial(t, `(>a+)ab`, 0, "aaab", "Fail.")
	runRegexTrial(t, `(?>a+)b`, 0, "aaab", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `([[:]+)`, 0, "a:[b]:", "Pass. Group[0]=(1,2) Group[1]=(1,2)")
	runRegexTrial(t, `([[=]+)`, 0, "a=[b]=", "Pass. Group[0]=(1,2) Group[1]=(1,2)")
	runRegexTrial(t, `([[.]+)`, 0, "a.[b].", "Pass. Group[0]=(1,2) Group[1]=(1,2)")
	runRegexTrial(t, `[a[:]b[:c]`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `[a[:]b[:c]`, 0, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `((?>a+)b)`, 0, "aaab", "Pass. Group[0]=(0,4) Group[1]=(0,4)")
	runRegexTrial(t, `(?>(a+))b`, 0, "aaab", "Pass. Group[0]=(0,4) Group[1]=(0,3)")
	runRegexTrial(t, `((?>[^()]+)|\([^()]*\))+`, 0, "((abc(ade)ufh()()x", "Pass. Group[0]=(2,16) Group[1]=(2,3)(5,5)(10,3)(13,2)(15,2)(17,1)")
	runRegexTrial(t, `(?<=x+)`, 0, "xxxxy", "Pass. Group[0]=(1,0)")
	runRegexTrial(t, `a{37,17}`, 0, "-", "Error.")
	runRegexTrial(t, `\Z`, 0, "a\nb\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, 0, "a\nb\n", "Pass. Group[0]=(4,0)")
	runRegexTrial(t, `$`, 0, "a\nb\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\Z`, 0, "b\na\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, 0, "b\na\n", "Pass. Group[0]=(4,0)")
	runRegexTrial(t, `$`, 0, "b\na\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\Z`, 0, "b\na", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, 0, "b\na", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `$`, 0, "b\na", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\Z`, Multiline, "a\nb\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, Multiline, "a\nb\n", "Pass. Group[0]=(4,0)")
	runRegexTrial(t, `$`, Multiline, "a\nb\n", "Pass. Group[0]=(1,0)")
	runRegexTrial(t, `\Z`, Multiline, "b\na\n", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, Multiline, "b\na\n", "Pass. Group[0]=(4,0)")
	runRegexTrial(t, `$`, Multiline, "b\na\n", "Pass. Group[0]=(1,0)")
	runRegexTrial(t, `\Z`, Multiline, "b\na", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `\z`, Multiline, "b\na", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `$`, Multiline, "b\na", "Pass. Group[0]=(1,0)")
	runRegexTrial(t, `a\Z`, 0, "a\nb\n", "Fail.")
	runRegexTrial(t, `a\z`, 0, "a\nb\n", "Fail.")
	runRegexTrial(t, `a$`, 0, "a\nb\n", "Fail.")
	runRegexTrial(t, `a\Z`, 0, "b\na\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\z`, 0, "b\na\n", "Fail.")
	runRegexTrial(t, `a$`, 0, "b\na\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\Z`, 0, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\z`, 0, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a$`, 0, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\z`, Multiline, "a\nb\n", "Fail.")
	runRegexTrial(t, `a$`, Multiline, "a\nb\n", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a\Z`, Multiline, "b\na\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\z`, Multiline, "b\na\n", "Fail.")
	runRegexTrial(t, `a$`, Multiline, "b\na\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\Z`, Multiline, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a\z`, Multiline, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `a$`, Multiline, "b\na", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `aa\Z`, 0, "aa\nb\n", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "aa\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "aa\nb\n", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "b\naa\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\z`, 0, "b\naa\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "b\naa\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\Z`, 0, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\z`, 0, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa$`, 0, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\z`, Multiline, "aa\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "aa\nb\n", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `aa\Z`, Multiline, "b\naa\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\z`, Multiline, "b\naa\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "b\naa\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\Z`, Multiline, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\z`, Multiline, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa$`, Multiline, "b\naa", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `aa\Z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `aa$`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa$`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa\Z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `aa\z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `aa$`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `aa\Z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `aa\z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `aa$`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "ab\nb\n", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "ab\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "ab\nb\n", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "b\nab\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\z`, 0, "b\nab\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "b\nab\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\Z`, 0, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\z`, 0, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab$`, 0, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\z`, Multiline, "ab\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "ab\nb\n", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nab\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\z`, Multiline, "b\nab\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "b\nab\n", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\z`, Multiline, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab$`, Multiline, "b\nab", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `ab\Z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `ab$`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab$`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab\Z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `ab\z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `ab$`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `ab\Z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `ab\z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `ab$`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "abb\nb\n", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "abb\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "abb\nb\n", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "b\nabb\n", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\z`, 0, "b\nabb\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "b\nabb\n", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\Z`, 0, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\z`, 0, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb$`, 0, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\z`, Multiline, "abb\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "abb\nb\n", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nabb\n", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\z`, Multiline, "b\nabb\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "b\nabb\n", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\z`, Multiline, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb$`, Multiline, "b\nabb", "Pass. Group[0]=(2,3)")
	runRegexTrial(t, `abb\Z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `abb$`, 0, "b\nac", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "ac\nb\n", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "b\nac\n", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "b\nac", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb$`, 0, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb\Z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `abb\z`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `abb$`, 0, "b\nca", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "ca\nb\n", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "b\nca\n", "Fail.")
	runRegexTrial(t, `abb\Z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `abb\z`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `abb$`, Multiline, "b\nca", "Fail.")
	runRegexTrial(t, `(^|x)(c)`, 0, "ca", "Pass. Group[0]=(0,1) Group[1]=(0,0) Group[2]=(0,1)")
	runRegexTrial(t, `a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz`, 0, "x", "Fail.")
	runRegexTrial(t, `round\(((?>[^()]+))\)`, 0, "_I(round(xs * sz),1)", "Pass. Group[0]=(3,14) Group[1]=(9,7)")
	runRegexTrial(t, `foo.bart`, 0, "foo.bart", "Pass. Group[0]=(0,8)")
	runRegexTrial(t, `^d[x][x][x]`, Multiline, "abcd\ndxxx", "Pass. Group[0]=(5,4)")
	runRegexTrial(t, `.X(.+)+X`, 0, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)")
	runRegexTrial(t, `.X(.+)+XX`, 0, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)")
	runRegexTrial(t, `.XX(.+)+X`, 0, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)")
	runRegexTrial(t, `.X(.+)+X`, 0, "bbbbXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.X(.+)+XX`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.XX(.+)+X`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.X(.+)+[X]`, 0, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)")
	runRegexTrial(t, `.X(.+)+[X][X]`, 0, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)")
	runRegexTrial(t, `.XX(.+)+[X]`, 0, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)")
	runRegexTrial(t, `.X(.+)+[X]`, 0, "bbbbXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.X(.+)+[X][X]`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.XX(.+)+[X]`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.[X](.+)+[X]`, 0, "bbbbXcXaaaaaaaa", "Pass. Group[0]=(3,4) Group[1]=(5,1)")
	runRegexTrial(t, `.[X](.+)+[X][X]`, 0, "bbbbXcXXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(5,1)")
	runRegexTrial(t, `.[X][X](.+)+[X]`, 0, "bbbbXXcXaaaaaaaa", "Pass. Group[0]=(3,5) Group[1]=(6,1)")
	runRegexTrial(t, `.[X](.+)+[X]`, 0, "bbbbXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.[X](.+)+[X][X]`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `.[X][X](.+)+[X]`, 0, "bbbbXXXaaaaaaaaa", "Fail.")
	runRegexTrial(t, `tt+$`, 0, "xxxtt", "Pass. Group[0]=(3,2)")
	runRegexTrial(t, `([\d-z]+)`, 0, "a0-za", "Pass. Group[0]=(1,3) Group[1]=(1,3)")
	runRegexTrial(t, `([\d-\s]+)`, 0, "a0- z", "Pass. Group[0]=(1,3) Group[1]=(1,3)")
	runRegexTrial(t, `\GX.*X`, 0, "aaaXbX", "Fail.")
	runRegexTrial(t, `(\d+\.\d+)`, 0, "3.1415926", "Pass. Group[0]=(0,9) Group[1]=(0,9)")
	runRegexTrial(t, `(\ba.{0,10}br)`, 0, "have a web browser", "Pass. Group[0]=(5,8) Group[1]=(5,8)")
	runRegexTrial(t, `\.c(pp|xx|c)?$`, IgnoreCase, "Changes", "Fail.")
	runRegexTrial(t, `\.c(pp|xx|c)?$`, IgnoreCase, "IO.c", "Pass. Group[0]=(2,2) Group[1]=")
	runRegexTrial(t, `(\.c(pp|xx|c)?$)`, IgnoreCase, "IO.c", "Pass. Group[0]=(2,2) Group[1]=(2,2) Group[2]=")
	runRegexTrial(t, `^([a-z]:)`, 0, "C:/", "Fail.")
	runRegexTrial(t, `^\S\s+aa$`, Multiline, "\nx aa", "Pass. Group[0]=(1,4)")
	runRegexTrial(t, `(^|a)b`, 0, "ab", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `^([ab]*?)(b)?(c)$`, 0, "abac", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]= Group[3]=(3,1)")
	runRegexTrial(t, `(\w)?(abc)\1b`, 0, "abcab", "Fail.")
	runRegexTrial(t, `^(?:.,){2}c`, 0, "a,b,c", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `^(.,){2}c`, 0, "a,b,c", "Pass. Group[0]=(0,5) Group[1]=(0,2)(2,2)")
	runRegexTrial(t, `^(?:[^,]*,){2}c`, 0, "a,b,c", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `^([^,]*,){2}c`, 0, "a,b,c", "Pass. Group[0]=(0,5) Group[1]=(0,2)(2,2)")
	runRegexTrial(t, `^([^,]*,){3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]*,){3,}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]*,){0,3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,3},){3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,3},){3,}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,3},){0,3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,},){3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,},){3,}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{1,},){0,3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{0,3},){3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{0,3},){3,}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `^([^,]{0,3},){0,3}d`, 0, "aaa,b,c,d", "Pass. Group[0]=(0,9) Group[1]=(0,4)(4,2)(6,2)")
	runRegexTrial(t, `(?i)`, 0, "", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `(?!\A)x`, Multiline, "a\nxb\n", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `^(a(b)?)+$`, 0, "aba", "Pass. Group[0]=(0,3) Group[1]=(0,2)(2,1) Group[2]=(1,1)")
	runRegexTrial(t, `^(aa(bb)?)+$`, 0, "aabbaa", "Pass. Group[0]=(0,6) Group[1]=(0,4)(4,2) Group[2]=(2,2)")
	runRegexTrial(t, `^.{9}abc.*\n`, Multiline, "123\nabcabcabcabc\n", "Pass. Group[0]=(4,13)")
	runRegexTrial(t, `^(a)?a$`, 0, "a", "Pass. Group[0]=(0,1) Group[1]=")
	runRegexTrial(t, `^(a)?(?(1)a|b)+$`, 0, "a", "Fail.")
	runRegexTrial(t, `^(a\1?)(a\1?)(a\2?)(a\3?)$`, 0, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(0,1) Group[2]=(1,2) Group[3]=(3,1) Group[4]=(4,2)")
	runRegexTrial(t, `^(a\1?){4}$`, 0, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(0,1)(1,2)(3,1)(4,2)")
	runRegexTrial(t, `^(0+)?(?:x(1))?`, 0, "x1", "Pass. Group[0]=(0,2) Group[1]= Group[2]=(1,1)")
	runRegexTrial(t, `^([0-9a-fA-F]+)(?:x([0-9a-fA-F]+)?)(?:x([0-9a-fA-F]+))?`, 0, "012cxx0190", "Pass. Group[0]=(0,10) Group[1]=(0,4) Group[2]= Group[3]=(6,4)")
	runRegexTrial(t, `^(b+?|a){1,2}c`, 0, "bbbac", "Pass. Group[0]=(0,5) Group[1]=(0,3)(3,1)")
	runRegexTrial(t, `^(b+?|a){1,2}c`, 0, "bbbbac", "Pass. Group[0]=(0,6) Group[1]=(0,4)(4,1)")
	runRegexTrial(t, `\((\w\. \w+)\)`, 0, "cd. (A. Tw)", "Pass. Group[0]=(4,7) Group[1]=(5,5)")
	runRegexTrial(t, `((?:aaaa|bbbb)cccc)?`, 0, "aaaacccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)")
	runRegexTrial(t, `((?:aaaa|bbbb)cccc)?`, 0, "bbbbcccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)")

	runRegexTrial(t, `^(foo)|(bar)$`, 0, "foobar", "Pass. Group[0]=(0,3) Group[1]=(0,3) Group[2]=")
	runRegexTrial(t, `^(foo)|(bar)$`, RightToLeft, "foobar", "Pass. Group[0]=(3,3) Group[1]= Group[2]=(3,3)")

	runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")
	runRegexTrial(t, `bab`, RightToLeft, "babababaa", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `abb`, RightToLeft, "abb", "Pass. Group[0]=(0,3)")

	runRegexTrial(t, `b$`, RightToLeft|Multiline, "aab\naab", "Pass. Group[0]=(6,1)")
	runRegexTrial(t, `^a`, RightToLeft|Multiline, "aab\naab", "Pass. Group[0]=(4,1)")
	runRegexTrial(t, `^aaab`, RightToLeft|Multiline, "aaab\naab", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `abb{2}`, RightToLeft, "abbb", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `abb{1,2}`, RightToLeft, "abbb", "Pass. Group[0]=(0,4)")

	runRegexTrial(t, `abb{1,2}`, RightToLeft, "abbbbbaaaa", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `\Ab`, RightToLeft, "bab\naaa", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\Abab$`, RightToLeft, "bab", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `b\Z`, RightToLeft, "bab\naaa", "Fail.")
	runRegexTrial(t, `b\Z`, RightToLeft, "babaaab", "Pass. Group[0]=(6,1)")
	runRegexTrial(t, `b\z`, RightToLeft, "babaaa", "Fail.")
	runRegexTrial(t, `b\z`, RightToLeft, "babaaab", "Pass. Group[0]=(6,1)")
	runRegexTrial(t, `a\G`, RightToLeft, "babaaa", "Pass. Group[0]=(5,1)")
	runRegexTrial(t, `\Abaaa\G`, RightToLeft, "baaa", "Pass. Group[0]=(0,4)")
	//			runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")
	//			runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")
	//			runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")
	//			runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")

	runRegexTrial(t, `\bc`, RightToLeft, "aaa c aaa c a", "Pass. Group[0]=(10,1)")
	runRegexTrial(t, `\bc`, RightToLeft, "c aaa c", "Pass. Group[0]=(6,1)")
	runRegexTrial(t, `\bc`, RightToLeft, "aaa ac", "Fail.")
	runRegexTrial(t, `\bc`, RightToLeft, "c aaa", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\bc`, RightToLeft, "aaacaaa", "Fail.")
	runRegexTrial(t, `\bc`, RightToLeft, "aaac aaa", "Fail.")
	runRegexTrial(t, `\bc`, RightToLeft, "aaa ca caaa", "Pass. Group[0]=(7,1)")

	runRegexTrial(t, `\Bc`, RightToLeft, "ac aaa ac", "Pass. Group[0]=(8,1)")
	runRegexTrial(t, `\Bc`, RightToLeft, "aaa c", "Fail.")
	runRegexTrial(t, `\Bc`, RightToLeft, "ca aaa", "Fail.")
	runRegexTrial(t, `\Bc`, RightToLeft, "aaa c aaa", "Fail.")
	runRegexTrial(t, `\Bc`, RightToLeft, " acaca ", "Pass. Group[0]=(4,1)")
	runRegexTrial(t, `\Bc`, RightToLeft, "aaac aaac", "Pass. Group[0]=(8,1)")
	runRegexTrial(t, `\Bc`, RightToLeft, "aaa caaa", "Fail.")

	runRegexTrial(t, `b(a?)b`, RightToLeft, "aabababbaaababa", "Pass. Group[0]=(11,3) Group[1]=(12,1)")
	runRegexTrial(t, `b{4}`, RightToLeft, "abbbbaabbbbaabbb", "Pass. Group[0]=(7,4)")
	runRegexTrial(t, `b\1aa(.)`, RightToLeft, "bBaaB", "Pass. Group[0]=(0,5) Group[1]=(4,1)")
	runRegexTrial(t, `b(.)aa\1`, RightToLeft, "bBaaB", "Fail.")

	runRegexTrial(t, `^(a\1?){4}$`, RightToLeft, "aaaaaa", "Pass. Group[0]=(0,6) Group[1]=(5,1)(3,2)(2,1)(0,2)")
	runRegexTrial(t, `^([0-9a-fA-F]+)(?:x([0-9a-fA-F]+)?)(?:x([0-9a-fA-F]+))?`, RightToLeft, "012cxx0190", "Pass. Group[0]=(0,10) Group[1]=(0,4) Group[2]= Group[3]=(6,4)")
	runRegexTrial(t, `^(b+?|a){1,2}c`, RightToLeft, "bbbac", "Pass. Group[0]=(0,5) Group[1]=(3,1)(0,3)")
	runRegexTrial(t, `\((\w\. \w+)\)`, RightToLeft, "cd. (A. Tw)", "Pass. Group[0]=(4,7) Group[1]=(5,5)")
	runRegexTrial(t, `((?:aaaa|bbbb)cccc)?`, RightToLeft, "aaaacccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)")
	runRegexTrial(t, `((?:aaaa|bbbb)cccc)?`, RightToLeft, "bbbbcccc", "Pass. Group[0]=(0,8) Group[1]=(0,8)")

	runRegexTrial(t, `(?<=a)b`, RightToLeft, "ab", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(?<=a)b`, RightToLeft, "cb", "Fail.")
	runRegexTrial(t, `(?<=a)b`, RightToLeft, "b", "Fail.")
	runRegexTrial(t, `(?<!c)b`, RightToLeft, "ab", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `(?<!c)b`, RightToLeft, "cb", "Fail.")
	runRegexTrial(t, `(?<!c)b`, RightToLeft, "b", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(?<!c)b`, RightToLeft, "b", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a(?=d).`, RightToLeft, "adabad", "Pass. Group[0]=(4,2)")
	runRegexTrial(t, `a(?=c|d).`, RightToLeft, "adabad", "Pass. Group[0]=(4,2)")

	runRegexTrial(t, `ab*c`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab*bc`, RightToLeft, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab*bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `.{1}`, RightToLeft, "abbbbc", "Pass. Group[0]=(5,1)")
	runRegexTrial(t, `.{3,4}`, RightToLeft, "abbbbc", "Pass. Group[0]=(2,4)")
	runRegexTrial(t, `ab{0,}bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab+bc`, RightToLeft, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab+bc`, RightToLeft, "abc", "Fail.")
	runRegexTrial(t, `ab+bc`, RightToLeft, "abq", "Fail.")
	runRegexTrial(t, `ab{1,}bc`, RightToLeft, "abq", "Fail.")
	runRegexTrial(t, `ab+bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,}bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{1,3}bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{3,4}bc`, RightToLeft, "abbbbc", "Pass. Group[0]=(0,6)")
	runRegexTrial(t, `ab{4,5}bc`, RightToLeft, "abbbbc", "Fail.")
	runRegexTrial(t, `ab?bc`, RightToLeft, "abbc", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `ab?bc`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}bc`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab?bc`, RightToLeft, "abbbbc", "Fail.")
	runRegexTrial(t, `ab?c`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `ab{0,1}c`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, RightToLeft, "abcc", "Fail.")
	runRegexTrial(t, `^abc`, RightToLeft, "abcc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^abc$`, RightToLeft, "aabc", "Fail.")
	runRegexTrial(t, `abc$`, RightToLeft, "aabc", "Pass. Group[0]=(1,3)")
	runRegexTrial(t, `abc$`, RightToLeft, "aabcd", "Fail.")
	runRegexTrial(t, `^`, RightToLeft, "abc", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `$`, RightToLeft, "abc", "Pass. Group[0]=(3,0)")
	runRegexTrial(t, `a.c`, RightToLeft, "abc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.c`, RightToLeft, "axc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a.*c`, RightToLeft, "axyzc", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `a.*c`, RightToLeft, "axyzd", "Fail.")
	runRegexTrial(t, `a[bc]d`, RightToLeft, "abc", "Fail.")
	runRegexTrial(t, `a[bc]d`, RightToLeft, "abd", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]e`, RightToLeft, "abd", "Fail.")
	runRegexTrial(t, `a[b-d]e`, RightToLeft, "ace", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[b-d]`, RightToLeft, "aac", "Pass. Group[0]=(1,2)")
	runRegexTrial(t, `a[-b]`, RightToLeft, "a-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-]`, RightToLeft, "a-", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[b-a]`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `a[]b`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `a[`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `a]`, RightToLeft, "a]", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a[]]b`, RightToLeft, "a]b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, RightToLeft, "aed", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^bc]d`, RightToLeft, "abd", "Fail.")
	runRegexTrial(t, `a[^-b]c`, RightToLeft, "adc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[^-b]c`, RightToLeft, "a-c", "Fail.")
	runRegexTrial(t, `a[^]b]c`, RightToLeft, "a]c", "Fail.")
	runRegexTrial(t, `a[^]b]c`, RightToLeft, "adc", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `\ba\b`, RightToLeft, "a-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\ba\b`, RightToLeft, "-a", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\ba\b`, RightToLeft, "-a-", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\by\b`, RightToLeft, "xy", "Fail.")
	runRegexTrial(t, `\by\b`, RightToLeft, "yz", "Fail.")
	runRegexTrial(t, `\by\b`, RightToLeft, "xyz", "Fail.")
	runRegexTrial(t, `\Ba\B`, RightToLeft, "a-", "Fail.")
	runRegexTrial(t, `\Ba\B`, RightToLeft, "-a", "Fail.")
	runRegexTrial(t, `\Ba\B`, RightToLeft, "-a-", "Fail.")
	runRegexTrial(t, `\By\b`, RightToLeft, "xy", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\by\B`, RightToLeft, "yz", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\By\B`, RightToLeft, "xyz", "Pass. Group[0]=(1,1)")
	runRegexTrial(t, `\w`, RightToLeft, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\w`, RightToLeft, "-", "Fail.")
	runRegexTrial(t, `\W`, RightToLeft, "a", "Fail.")
	runRegexTrial(t, `\W`, RightToLeft, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a\sb`, RightToLeft, "a b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a\sb`, RightToLeft, "a-b", "Fail.")
	runRegexTrial(t, `a\Sb`, RightToLeft, "a b", "Fail.")
	runRegexTrial(t, `a\Sb`, RightToLeft, "a-b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `\d`, RightToLeft, "1", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `\d`, RightToLeft, "-", "Fail.")
	runRegexTrial(t, `\D`, RightToLeft, "1", "Fail.")
	runRegexTrial(t, `\D`, RightToLeft, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\w]`, RightToLeft, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\w]`, RightToLeft, "-", "Fail.")
	runRegexTrial(t, `[\W]`, RightToLeft, "a", "Fail.")
	runRegexTrial(t, `[\W]`, RightToLeft, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `a[\s]b`, RightToLeft, "a b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a[\s]b`, RightToLeft, "a-b", "Fail.")
	runRegexTrial(t, `a[\S]b`, RightToLeft, "a b", "Fail.")
	runRegexTrial(t, `a[\S]b`, RightToLeft, "a-b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `[\d]`, RightToLeft, "1", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `[\d]`, RightToLeft, "-", "Fail.")
	runRegexTrial(t, `[\D]`, RightToLeft, "1", "Fail.")
	runRegexTrial(t, `[\D]`, RightToLeft, "-", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `ab|cd`, RightToLeft, "abc", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `ab|cd`, RightToLeft, "abcd", "Pass. Group[0]=(2,2)")
	runRegexTrial(t, `()ef`, RightToLeft, "def", "Pass. Group[0]=(1,2) Group[1]=(1,0)")
	runRegexTrial(t, `*a`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `(*)b`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `$b`, RightToLeft, "b", "Fail.")
	runRegexTrial(t, `a\`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `a\(b`, RightToLeft, "a(b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `a\(*b`, RightToLeft, "ab", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `a\(*b`, RightToLeft, "a((b", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a\\b`, RightToLeft, "a\\b", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc)`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `(abc`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `((a))`, RightToLeft, "abc", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1)")
	runRegexTrial(t, `(a)b(c)`, RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)")
	runRegexTrial(t, `a+b+c`, RightToLeft, "aabbabc", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a{1,}b{1,}c`, RightToLeft, "aabbabc", "Pass. Group[0]=(4,3)")
	runRegexTrial(t, `a**`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `a.+?c`, RightToLeft, "abcabc", "Pass. Group[0]=(3,3)")
	runRegexTrial(t, `(a+|b)*`, RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)")
	runRegexTrial(t, `(a+|b){0,}`, RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)")
	runRegexTrial(t, `(a+|b)+`, RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)")
	runRegexTrial(t, `(a+|b){1,}`, RightToLeft, "ab", "Pass. Group[0]=(0,2) Group[1]=(1,1)(0,1)")
	runRegexTrial(t, `(a+|b)?`, RightToLeft, "ab", "Pass. Group[0]=(1,1) Group[1]=(1,1)")
	runRegexTrial(t, `(a+|b){0,1}`, RightToLeft, "ab", "Pass. Group[0]=(1,1) Group[1]=(1,1)")
	runRegexTrial(t, `)(`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `[^ab]*`, RightToLeft, "cde", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `abc`, RightToLeft, "", "Fail.")
	runRegexTrial(t, `a*`, RightToLeft, "", "Pass. Group[0]=(0,0)")
	runRegexTrial(t, `([abc])*d`, RightToLeft, "abbbcd", "Pass. Group[0]=(0,6) Group[1]=(4,1)(3,1)(2,1)(1,1)(0,1)")
	runRegexTrial(t, `([abc])*bcd`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,1)")
	runRegexTrial(t, `a|b|c|d|e`, RightToLeft, "e", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `(a|b|c|d|e)f`, RightToLeft, "ef", "Pass. Group[0]=(0,2) Group[1]=(0,1)")
	runRegexTrial(t, `abcd*efg`, RightToLeft, "abcdefg", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `ab*`, RightToLeft, "xabyabbbz", "Pass. Group[0]=(4,4)")
	runRegexTrial(t, `ab*`, RightToLeft, "xayabbbz", "Pass. Group[0]=(3,4)")
	runRegexTrial(t, `(ab|cd)e`, RightToLeft, "abcde", "Pass. Group[0]=(2,3) Group[1]=(2,2)")
	runRegexTrial(t, `[abhgefdc]ij`, RightToLeft, "hij", "Pass. Group[0]=(0,3)")
	runRegexTrial(t, `^(ab|cd)e`, RightToLeft, "abcde", "Fail.")
	runRegexTrial(t, `(abc|)ef`, RightToLeft, "abcdef", "Pass. Group[0]=(4,2) Group[1]=(4,0)")
	runRegexTrial(t, `(a|b)c*d`, RightToLeft, "abcd", "Pass. Group[0]=(1,3) Group[1]=(1,1)")
	runRegexTrial(t, `(ab|ab*)bc`, RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)")
	runRegexTrial(t, `a([bc]*)c*`, RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(1,1)")
	runRegexTrial(t, `a([bc]*)(c*d)`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)")
	runRegexTrial(t, `a([bc]+)(c*d)`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)")
	runRegexTrial(t, `a([bc]*)(c+d)`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,1) Group[2]=(2,2)")
	runRegexTrial(t, `a[bcd]*dcdcde`, RightToLeft, "adcdcde", "Pass. Group[0]=(0,7)")
	runRegexTrial(t, `a[bcd]+dcdcde`, RightToLeft, "adcdcde", "Fail.")
	runRegexTrial(t, `(ab|a)b*c`, RightToLeft, "abc", "Pass. Group[0]=(0,3) Group[1]=(0,1)")
	runRegexTrial(t, `((a)(b)c)(d)`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(0,3) Group[2]=(0,1) Group[3]=(1,1) Group[4]=(3,1)")
	runRegexTrial(t, `[a-zA-Z_][a-zA-Z0-9_]*`, RightToLeft, "alpha", "Pass. Group[0]=(0,5)")
	runRegexTrial(t, `^a(bc+|b[eh])g|.h$`, RightToLeft, "abh", "Pass. Group[0]=(1,2) Group[1]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, RightToLeft, "effgz", "Pass. Group[0]=(0,5) Group[1]=(0,5) Group[2]=")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, RightToLeft, "ij", "Pass. Group[0]=(0,2) Group[1]=(0,2) Group[2]=(1,1)")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, RightToLeft, "effg", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, RightToLeft, "bcdd", "Fail.")
	runRegexTrial(t, `(bc+d$|ef*g.|h?i(j|k))`, RightToLeft, "reffgz", "Pass. Group[0]=(1,5) Group[1]=(1,5) Group[2]=")
	runRegexTrial(t, `((((((((((a))))))))))`, RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `((((((((((a))))))))))\10`, RightToLeft, "aa", "Fail.")
	runRegexTrial(t, `\10((((((((((a))))))))))`, RightToLeft, "aa", "Pass. Group[0]=(0,2) Group[1]=(1,1) Group[2]=(1,1) Group[3]=(1,1) Group[4]=(1,1) Group[5]=(1,1) Group[6]=(1,1) Group[7]=(1,1) Group[8]=(1,1) Group[9]=(1,1) Group[10]=(1,1)")
	runRegexTrial(t, `((((((((((a))))))))))!`, RightToLeft, "aa", "Fail.")
	runRegexTrial(t, `((((((((((a))))))))))!`, RightToLeft, "a!", "Pass. Group[0]=(0,2) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1) Group[10]=(0,1)")
	runRegexTrial(t, `(((((((((a)))))))))`, RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1) Group[2]=(0,1) Group[3]=(0,1) Group[4]=(0,1) Group[5]=(0,1) Group[6]=(0,1) Group[7]=(0,1) Group[8]=(0,1) Group[9]=(0,1)")
	runRegexTrial(t, `multiple words of text`, RightToLeft, "uh-uh", "Fail.")
	runRegexTrial(t, `multiple words`, RightToLeft, "multiple words, yeah", "Pass. Group[0]=(0,14)")
	runRegexTrial(t, `(.*)c(.*)`, RightToLeft, "abcde", "Pass. Group[0]=(0,5) Group[1]=(0,2) Group[2]=(3,2)")
	runRegexTrial(t, `\((.*), (.*)\)`, RightToLeft, "(a, b)", "Pass. Group[0]=(0,6) Group[1]=(1,1) Group[2]=(4,1)")
	runRegexTrial(t, `[k]`, RightToLeft, "ab", "Fail.")
	runRegexTrial(t, `abcd`, RightToLeft, "abcd", "Pass. Group[0]=(0,4)")
	runRegexTrial(t, `a(bc)d`, RightToLeft, "abcd", "Pass. Group[0]=(0,4) Group[1]=(1,2)")
	runRegexTrial(t, `a[-]?c`, RightToLeft, "ac", "Pass. Group[0]=(0,2)")
	runRegexTrial(t, `(abc)\1`, RightToLeft, "abcabc", "Fail.")
	runRegexTrial(t, `\1(abc)`, RightToLeft, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(3,3)")
	runRegexTrial(t, `([a-c]*)\1`, RightToLeft, "abcabc", "Fail.")
	runRegexTrial(t, `\1([a-c]*)`, RightToLeft, "abcabc", "Pass. Group[0]=(0,6) Group[1]=(3,3)")
	runRegexTrial(t, `\1`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `\2`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `(a)|\1`, RightToLeft, "a", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `(a)|\1`, RightToLeft, "x", "Fail.")
	runRegexTrial(t, `(a)|\2`, RightToLeft, "-", "Error.")
	runRegexTrial(t, `(([a-c])b*?\2)*`, RightToLeft, "ababbbcbc", "Pass. Group[0]=(9,0) Group[1]= Group[2]=")
	runRegexTrial(t, `(([a-c])b*?\2){3}`, RightToLeft, "ababbbcbc", "Fail.")
	runRegexTrial(t, `((\3|b)\2(a)x)+`, RightToLeft, "aaxabxbaxbbx", "Fail.")
	runRegexTrial(t, `((\3|b)\2(a)x)+`, RightToLeft, "aaaxabaxbaaxbbax", "Fail.")
	runRegexTrial(t, `((\3|b)\2(a)){2,}`, RightToLeft, "bbaababbabaaaaabbaaaabba", "Fail.")

	runRegexTrial(t, `\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)`, 0, "((a(b))c)", "Pass. Group[0]=(0,9) Group[1]=")
	runRegexTrial(t, `^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$`, 0, "((a(b))c)", "Pass. Group[0]=(0,9) Group[1]=")
	runRegexTrial(t, `^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$`, 0, "((a(b))c", "Fail.")

	runRegexTrial(t, `^\((?>[^()]+|\((?<depth>)|\)(?<-depth>))*(?(depth)(?!))\)$`, 0, "())", "Fail.")

	runRegexTrial(t, `(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))`, 0, "((a(b))c)",
		"Pass. Group[0]=(0,9) Group[1]=(0,9) Group[2]=(0,1)(1,2)(3,2) Group[3]=(5,1)(6,2)(8,1) Group[4]= Group[5]=(4,1)(2,4)(1,7)")
	runRegexTrial(t, `^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$`, 0, "((a(b))c)",
		"Pass. Group[0]=(0,9) Group[1]=(0,9) Group[2]=(0,1)(1,2)(3,2) Group[3]=(5,1)(6,2)(8,1) Group[4]= Group[5]=(4,1)(2,4)(1,7)")
	runRegexTrial(t, `(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))`, 0, "x(a((b)))b)x",
		"Pass. Group[0]=(1,9) Group[1]=(1,9) Group[2]=(1,2)(3,1)(4,2) Group[3]=(6,1)(7,1)(8,2) Group[4]= Group[5]=(5,1)(4,3)(2,6)")
	runRegexTrial(t, `(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))`, 0, "x((a((b)))x",
		"Pass. Group[0]=(2,9) Group[1]=(2,9) Group[2]=(2,2)(4,1)(5,2) Group[3]=(7,1)(8,1)(9,2) Group[4]= Group[5]=(6,1)(5,3)(3,6)")
	runRegexTrial(t, `^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$`, 0, "((a(b))c", "Fail.")
	runRegexTrial(t, `^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$`, 0, "((a(b))c))", "Fail.")
	runRegexTrial(t, `^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$`, 0, ")(", "Fail.")
	runRegexTrial(t, `^(((?<foo>\()[^()]*)+((?<bar-foo>\))[^()]*)+)+(?(foo)(?!))$`, 0, "((a((b))c)", "Fail.")

	runRegexTrial(t, `b`, RightToLeft, "babaaa", "Pass. Group[0]=(2,1)")

	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[n]", "Pass. Group[0]=(0,3) Group[1]=(1,1)")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "n", "Pass. Group[0]=(0,1) Group[1]=(0,1)")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "n[i]e", "Fail.")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[n", "Fail.")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "]n]", "Fail.")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, `\[n\]`, "Fail.")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, `[n\]`, "Pass. Group[0]=(0,4) Group[1]=(1,2)")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, `[n\[]`, "Pass. Group[0]=(0,5) Group[1]=(1,3)")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, `[[n]`, "Pass. Group[0]=(0,4) Group[1]=(1,2)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[s] . [n]", "Pass. Group[0]=(0,9) Group[1]=(1,1) Group[2]=(7,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[s] . n", "Pass. Group[0]=(0,7) Group[1]=(1,1) Group[2]=(6,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "s.[ n ]", "Pass. Group[0]=(0,7) Group[1]=(0,1) Group[2]=(3,3)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, " . n", "Pass. Group[0]=(0,4) Group[1]=(0,1) Group[2]=(3,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "s. ", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[.]. ", "Pass. Group[0]=(0,5) Group[1]=(1,1) Group[2]=(4,1)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[c].[s].[n]", "Pass. Group[0]=(0,11) Group[1]=(1,1) Group[2]=(5,1) Group[3]=(9,1)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, " c . s . n ", "Pass. Group[0]=(0,11) Group[1]=(0,3) Group[2]=(5,2) Group[3]=(9,2)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, " . [.] . [ ]", "Pass. Group[0]=(0,12) Group[1]=(0,1) Group[2]=(4,1) Group[3]=(10,1)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "c.n", "Pass. Group[0]=(0,3) Group[1]=(0,1) Group[2]=(2,1)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[c] .[n]", "Pass. Group[0]=(0,8) Group[1]=(1,1) Group[2]=(6,1)")
	runRegexTrial(t, `^((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "c.n.", "Fail.")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "s.c.n", "Pass. Group[0]=(0,5) Group[1]=(0,1) Group[2]=(2,1) Group[3]=(4,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[s].[c].[n]", "Pass. Group[0]=(0,11) Group[1]=(1,1) Group[2]=(5,1) Group[3]=(9,1)")
	runRegexTrial(t, `^((\[(?<SCHEMA>[^\]]+)\])|(?<SCHEMA>[^\.\[\]]+))\s*\.\s*((\[(?<CATALOG>[^\]]+)\])|(?<CATALOG>[^\.\[\]]+))\s*\.\s*((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, Compiled|IgnoreCase|ExplicitCapture, "[s].[c].", "Fail.")
	runRegexTrial(t, `^((\[(?<ColName>.+)\])|(?<ColName>\S+))([ ]+(?<Order>ASC|DESC))?$`, IgnoreCase|ExplicitCapture, "[id]]", "Pass. Group[0]=(0,5) Group[1]=(1,3) Group[2]=")
	runRegexTrial(t, `a{1,2147483647}`, 0, "a", "Pass. Group[0]=(0,1)")
	runRegexTrial(t, `^((\[(?<NAME>[^\]]+)\])|(?<NAME>[^\.\[\]]+))$`, 0, "[a]", "Pass. Group[0]=(0,3) Group[1]=(0,3) Group[2]=(0,3) Group[3]=(1,1)")

}

func runRegexTrial(t *testing.T, pattern string, options RegexOptions, input, expected string) {
	result := ""

	re, err := Compile(pattern, options)
	if err != nil {
		if expected != "Error." {
			t.Errorf("Compiling pattern '%v' with options '%v' -- expected '%v' got '%v'", pattern, options, expected, err.Error())
		}
		return
	}

	groupNums := re.GetGroupNumbers()
	m, err := re.FindStringMatch(input)
	if err != nil {
		result = fmt.Sprintf("Error. %v", err.Error())
	} else if m != nil {
		result = "Pass."
		for i, group := range m.Groups() {
			gid := groupNums[i]

			result += fmt.Sprintf(" Group[%v]=", gid)
			for _, cp := range group.Captures {
				result += fmt.Sprintf("(%v,%v)", cp.Index, cp.Length)
			}
		}
	} else {
		result = "Fail."
	}

	if expected != result {
		t.Errorf("Matching input '%v' against pattern '%v' with options '%v' -- expected '%v' got '%v'", input, pattern, options, expected, result)
	}
}
