/**
 * @fileoverview Test cases for the tick-generating functions.
 * These were generated by adding logging code to the old ticker functions. The
 * tests serve to track existing behavior should it change in the future.
 *
 * @author danvdk@gmail.com (Dan Vanderkam)
 */

import * as DygraphTickers from '../../src/dygraph-tickers';
import DEFAULT_ATTRS from '../../src/dygraph-default-attrs';

describe("date-ticker-tests", function() {

cleanupAfterEach();

var createOptionsViewForAxis = function(axis, dict) {
  return function (x) {
    if (dict && dict.hasOwnProperty(x)) {
      return dict[x];
    }
    if (DEFAULT_ATTRS.axes[axis].hasOwnProperty(x)) {
      return DEFAULT_ATTRS.axes[axis][x];
    }
    if (DEFAULT_ATTRS.hasOwnProperty(x)) {
      return DEFAULT_ATTRS[x];
    }
    if (x == 'axisLabelFormatter') return null;
    throw "mysterious " + axis + "-axis option: " + x;
  };
};

// Change '&#160;' (non-breaking space) to ' ' in all labels. Cleans up expected values.
function changeNbspToSpace(ticks) {
  for (var i = 0; i < ticks.length; i++) {
    if (ticks[i].label) {
      ticks[i].label = ticks[i].label.replace(/&#160;/g, ' ');
    }
  }
}

it('testBasicDateTicker', function() {
  var opts = {labelsUTC: true};
  var options = createOptionsViewForAxis('x', opts);

  var ticks = DygraphTickers.dateTicker(-1797534000000, 1255579200000, 800, options);
  var expected_ticks = [
      {"v":-1577923200000,"label":"1920"},
      {"v":-1262304000000,"label":"1930"},
      {"v":-946771200000, "label":"1940"},
      {"v":-631152000000, "label":"1950"},
      {"v":-315619200000, "label":"1960"},
      {"v": 0,            "label":"1970"},
      {"v": 315532800000, "label":"1980"},
      {"v": 631152000000, "label":"1990"},
      {"v": 946684800000, "label":"2000"}
  ];
  assert.deepEqual(expected_ticks, ticks);

  var start = Date.UTC(1999, 11, 31, 14, 0, 0);
  var end = Date.UTC(2000,  0,  1, 12, 0, 0);
  var granularity = DygraphTickers.Granularity.TWO_HOURLY;
  ticks = DygraphTickers.getDateAxis(start, end, granularity, options);
  changeNbspToSpace(ticks);
  expected_ticks = [ // months of the year are zero-based.
      {v: Date.UTC(1999, 11, 31, 14, 0, 0), label: '14:00'},
      {v: Date.UTC(1999, 11, 31, 16, 0, 0), label: '16:00'},
      {v: Date.UTC(1999, 11, 31, 18, 0, 0), label: '18:00'},
      {v: Date.UTC(1999, 11, 31, 20, 0, 0), label: '20:00'},
      {v: Date.UTC(1999, 11, 31, 22, 0, 0), label: '22:00'},
      {v: Date.UTC(2000,  0,  1,  0, 0, 0), label: '01 Jan'},
      {v: Date.UTC(2000,  0,  1,  2, 0, 0), label: '02:00'},
      {v: Date.UTC(2000,  0,  1,  4, 0, 0), label: '04:00'},
      {v: Date.UTC(2000,  0,  1,  6, 0, 0), label: '06:00'},
      {v: Date.UTC(2000,  0,  1,  8, 0, 0), label: '08:00'},
      {v: Date.UTC(2000,  0,  1, 10, 0, 0), label: '10:00'},
      {v: Date.UTC(2000,  0,  1, 12, 0, 0), label: '12:00'}
  ];
  assert.deepEqual(expected_ticks, ticks);
});

it('testAllDateTickers', function() {
  var opts = {labelsUTC: true, pixelsPerLabel: 60};
  var options = createOptionsViewForAxis('x', opts);

  // For granularities finer than MONTHLY, the first tick returned tick
  // could lie outside [start_time, end_time] range in the original code.
  // In these tests, those spurious ticks are removed to test new behavior.

  var ticker = function() {
    var ticks = DygraphTickers.dateTicker.apply(null, arguments);
    changeNbspToSpace(ticks);
    return ticks;
  };

  assert.deepEqual([{"v":-1577923200000,"label":"1920"},{"v":-1262304000000,"label":"1930"},{"v":-946771200000,"label":"1940"},{"v":-631152000000,"label":"1950"},{"v":-315619200000,"label":"1960"},{"v":0,"label":"1970"},{"v":315532800000,"label":"1980"},{"v":631152000000,"label":"1990"},{"v":946684800000,"label":"2000"}], ticker(-1797552000000, 1255561200000, 800, options));
  assert.deepEqual([{"v":-5364662400000,"label":"1800"},{"v":-2208988800000,"label":"1900"}], ticker(-6122044800000, 189302400000, 480, options));
  assert.deepEqual([{"v":1041120000000,"label":"29 Dec"},{"v":1041724800000,"label":"05 Jan"},{"v":1042329600000,"label":"12 Jan"},{"v":1042934400000,"label":"19 Jan"},{"v":1043539200000,"label":"26 Jan"},{"v":1044144000000,"label":"02 Feb"},{"v":1044748800000,"label":"09 Feb"},{"v":1045353600000,"label":"16 Feb"}], ticker(1041120000000, 1045353600000, 640, options));
  assert.deepEqual([{"v":1041379200000,"label":"Jan 2003"},{"v":1072915200000,"label":"Jan 2004"},{"v":1104537600000,"label":"Jan 2005"},{"v":1136073600000,"label":"Jan 2006"},{"v":1167609600000,"label":"Jan 2007"},{"v":1199145600000,"label":"Jan 2008"},{"v":1230768000000,"label":"Jan 2009"},{"v":1262304000000,"label":"Jan 2010"},{"v":1293840000000,"label":"Jan 2011"}], ticker(1041120000000, 1307833200000, 800, options));
  assert.deepEqual([{"v":1159660800000,"label":"01 Oct"},{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"}], ticker(1159657200000, 1162252800000, 480, options));
  assert.deepEqual([{"v":1159660800000,"label":"01 Oct"},{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"}], ticker(1159657200000, 1162252800000, 640, options));
  assert.deepEqual([{"v":1159660800000,"label":"01 Oct"},{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"},{"v":1162684800000,"label":"05 Nov"},{"v":1163289600000,"label":"12 Nov"},{"v":1163894400000,"label":"19 Nov"},{"v":1164499200000,"label":"26 Nov"}], ticker(1159657200000, 1164758400000, 1150, options));
  assert.deepEqual([{"v":1159660800000,"label":"Oct 2006"},{"v":1162339200000,"label":"Nov 2006"}], ticker(1159657200000, 1164758400000, 400, options));
  assert.deepEqual([{"v":1159660800000,"label":"01 Oct"},{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"},{"v":1162684800000,"label":"05 Nov"},{"v":1163289600000,"label":"12 Nov"},{"v":1163894400000,"label":"19 Nov"},{"v":1164499200000,"label":"26 Nov"}], ticker(1159657200000, 1164758400000, 500, options));
  assert.deepEqual([{"v":1159660800000,"label":"01 Oct"},{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"},{"v":1162684800000,"label":"05 Nov"},{"v":1163289600000,"label":"12 Nov"},{"v":1163894400000,"label":"19 Nov"},{"v":1164499200000,"label":"26 Nov"}], ticker(1159657200000, 1164758400000, 600, options));
  assert.deepEqual([{"v":1160265600000,"label":"08 Oct"},{"v":1160870400000,"label":"15 Oct"},{"v":1161475200000,"label":"22 Oct"},{"v":1162080000000,"label":"29 Oct"},{"v":1162684800000,"label":"05 Nov"},{"v":1163289600000,"label":"12 Nov"}], ticker(1160243979962, 1163887694248, 600, options));

  assert.deepEqual([{"v":1160611200000,"label":"12 Oct"},{"v":1160784000000,"label":"14 Oct"},{"v":1160956800000,"label":"16 Oct"},{"v":1161129600000,"label":"18 Oct"}], ticker(1160521200000, 1161298800000, 480, options));
  assert.deepEqual([{"v":1161475200000,"label":"22 Oct"},{"v":1161561600000,"label":"23 Oct"},{"v":1161648000000,"label":"24 Oct"},{"v":1161734400000,"label":"25 Oct"},{"v":1161820800000,"label":"26 Oct"},{"v":1161907200000,"label":"27 Oct"},{"v":1161993600000,"label":"28 Oct"}], ticker(1161471164461, 1161994065957, 600, options));
  assert.deepEqual([{"v":1161561600000,"label":"23 Oct"},{"v":1161583200000,"label":"06:00"},{"v":1161604800000,"label":"12:00"},{"v":1161626400000,"label":"18:00"}], ticker(1161557878860, 1161642991675, 600, options));
  assert.deepEqual([{"v":1161756000000,"label":"06:00"},{"v":1161759600000,"label":"07:00"},{"v":1161763200000,"label":"08:00"},{"v":1161766800000,"label":"09:00"},{"v":1161770400000,"label":"10:00"},{"v":1161774000000,"label":"11:00"},{"v":1161777600000,"label":"12:00"}], ticker(1161752537840, 1161777663332, 600, options));
  assert.deepEqual([{"v":1167609600000,"label":"01 Jan"},{"v":1167696000000,"label":"02 Jan"},{"v":1167782400000,"label":"03 Jan"},{"v":1167868800000,"label":"04 Jan"},{"v":1167955200000,"label":"05 Jan"},{"v":1168041600000,"label":"06 Jan"},{"v":1168128000000,"label":"07 Jan"},{"v":1168214400000,"label":"08 Jan"},{"v":1168300800000,"label":"09 Jan"}], ticker(1167609600000, 1168300800000, 480, options));
  assert.deepEqual([{"v":1167609600000,"label":"Jan 2007"}], ticker(1167609600000, 1199059200000, 100, options));
  assert.deepEqual([{"v":1167609600000,"label":"Jan 2007"},{"v":1175385600000,"label":"Apr 2007"},{"v":1183248000000,"label":"Jul 2007"},{"v":1191196800000,"label":"Oct 2007"}], ticker(1167609600000, 1199059200000, 300, options));
  assert.deepEqual([{"v":1167609600000,"label":"Jan 2007"},{"v":1175385600000,"label":"Apr 2007"},{"v":1183248000000,"label":"Jul 2007"},{"v":1191196800000,"label":"Oct 2007"}], ticker(1167609600000, 1199059200000, 480, options));
  assert.deepEqual([{"v":1167609600000,"label":"Jan 2007"},{"v":1175385600000,"label":"Apr 2007"},{"v":1183248000000,"label":"Jul 2007"},{"v":1191196800000,"label":"Oct 2007"}], ticker(1167609600000, 1199059200000, 600, options));
  assert.deepEqual([{"v":1160611200000,"label":"12 Oct"},{"v":1160784000000,"label":"14 Oct"},{"v":1160956800000,"label":"16 Oct"},{"v":1161129600000,"label":"18 Oct"}], ticker(1160521200000, 1161298800000, 480, options));
  assert.deepEqual([{"v":1167609600000,"label":"Jan 2007"},{"v":1170288000000,"label":"Feb 2007"},{"v":1172707200000,"label":"Mar 2007"},{"v":1175385600000,"label":"Apr 2007"},{"v":1177977600000,"label":"May 2007"},{"v":1180656000000,"label":"Jun 2007"},{"v":1183248000000,"label":"Jul 2007"},{"v":1185926400000,"label":"Aug 2007"},{"v":1188604800000,"label":"Sep 2007"},{"v":1191196800000,"label":"Oct 2007"},{"v":1193875200000,"label":"Nov 2007"},{"v":1196467200000,"label":"Dec 2007"}], ticker(1167609600000, 1199059200000, 800, options));

  assert.deepEqual([{"v":1293840000000,"label":"Jan 2011"},{"v":1296518400000,"label":"Feb 2011"},{"v":1298937600000,"label":"Mar 2011"},{"v":1301616000000,"label":"Apr 2011"},{"v":1304208000000,"label":"May 2011"},{"v":1306886400000,"label":"Jun 2011"},{"v":1309478400000,"label":"Jul 2011"},{"v":1312156800000,"label":"Aug 2011"}], ticker(1293753600000, 1312844400000, 727, options));
  assert.deepEqual([{"v":1201824000000,"label":"01 Feb"},{"v":1201910400000,"label":"02 Feb"},{"v":1201996800000,"label":"03 Feb"},{"v":1202083200000,"label":"04 Feb"},{"v":1202169600000,"label":"05 Feb"},{"v":1202256000000,"label":"06 Feb"}], ticker(1201824000000, 1202256000000, 700, options));
  assert.deepEqual([{"v":1210118400000,"label":"07 May"},{"v":1210140000000,"label":"06:00"},{"v":1210161600000,"label":"12:00"},{"v":1210183200000,"label":"18:00"},{"v":1210204800000,"label":"08 May"},{"v":1210226400000,"label":"06:00"},{"v":1210248000000,"label":"12:00"},{"v":1210269600000,"label":"18:00"},{"v":1210291200000,"label":"09 May"}], ticker(1210114800000, 1210291200000, 480, options));
  assert.deepEqual([{"v":1210118400000,"label":"07 May"},{"v":1210204800000,"label":"08 May"},{"v":1210291200000,"label":"09 May"},{"v":1210377600000,"label":"10 May"},{"v":1210464000000,"label":"11 May"}], ticker(1210114800000, 1210464000000, 480, options));
  assert.deepEqual([{"v":1210118400000,"label":"07 May"},{"v":1210204800000,"label":"08 May"},{"v":1210291200000,"label":"09 May"},{"v":1210377600000,"label":"10 May"},{"v":1210464000000,"label":"11 May"},{"v":1210550400000,"label":"12 May"}], ticker(1210114800000, 1210550400000, 480, options));
  assert.deepEqual([{"v":1214870400000,"label":"01 Jul"},{"v":1214872200000,"label":"00:30"},{"v":1214874000000,"label":"01:00"},{"v":1214875800000,"label":"01:30"}], ticker(1214870400000, 1214877599000, 600, options));
  assert.deepEqual([{"v":1214870400000,"label":"Jul 2008"},{"v":1217548800000,"label":"Aug 2008"},{"v":1220227200000,"label":"Sep 2008"}], ticker(1214866800000, 1222747200000, 600, options));
  assert.deepEqual([{"v":1215820800000,"label":"12 Jul"},{"v":1215842400000,"label":"06:00"},{"v":1215864000000,"label":"12:00"},{"v":1215885600000,"label":"18:00"},{"v":1215907200000,"label":"13 Jul"},{"v":1215928800000,"label":"06:00"},{"v":1215950400000,"label":"12:00"},{"v":1215972000000,"label":"18:00"}], ticker(1215817200000, 1215989940000, 600, options));
  assert.deepEqual([{"v":1246752000000,"label":"05 Jul"},{"v":1247356800000,"label":"12 Jul"},{"v":1247961600000,"label":"19 Jul"}], ticker(1246402800000, 1248217200000, 600, options));
  assert.deepEqual([{"v":1246752000000,"label":"05 Jul"},{"v":1247356800000,"label":"12 Jul"},{"v":1247961600000,"label":"19 Jul"},{"v":1248566400000,"label":"26 Jul"},{"v":1249171200000,"label":"02 Aug"}], ticker(1246402800000, 1249340400000, 600, options));
  assert.deepEqual([{"v":1247356800000,"label":"12 Jul"},{"v":1247360400000,"label":"01:00"},{"v":1247364000000,"label":"02:00"},{"v":1247367600000,"label":"03:00"},{"v":1247371200000,"label":"04:00"},{"v":1247374800000,"label":"05:00"},{"v":1247378400000,"label":"06:00"}], ticker(1247356800000, 1247378400000, 600, options));

  assert.deepEqual([{"v":1247356800000,"label":"12 Jul"},{"v":1247360400000,"label":"01:00"},{"v":1247364000000,"label":"02:00"},{"v":1247367600000,"label":"03:00"},{"v":1247371200000,"label":"04:00"},{"v":1247374800000,"label":"05:00"},{"v":1247378400000,"label":"06:00"}], ticker(1247356800000, 1247378400000, 600, options));
  assert.deepEqual([{"v":1254268800000,"label":"30 Sep"},{"v":1254355200000,"label":"01 Oct"},{"v":1254441600000,"label":"02 Oct"},{"v":1254528000000,"label":"03 Oct"},{"v":1254614400000,"label":"04 Oct"},{"v":1254700800000,"label":"05 Oct"},{"v":1254787200000,"label":"06 Oct"},{"v":1254873600000,"label":"07 Oct"},{"v":1254960000000,"label":"08 Oct"},{"v":1255046400000,"label":"09 Oct"},{"v":1255132800000,"label":"10 Oct"}], ticker(1254222000000, 1255172400000, 900, options));
  assert.deepEqual([{"v":1254441600000,"label":"02 Oct"},{"v":1254528000000,"label":"03 Oct"},{"v":1254614400000,"label":"04 Oct"},{"v":1254700800000,"label":"05 Oct"},{"v":1254787200000,"label":"06 Oct"},{"v":1254873600000,"label":"07 Oct"},{"v":1254960000000,"label":"08 Oct"}], ticker(1254394800000, 1254999600000, 900, options));
  assert.deepEqual([{"v":1259625600000,"label":"01 Dec"},{"v":1259712000000,"label":"02 Dec"},{"v":1259798400000,"label":"03 Dec"},{"v":1259884800000,"label":"04 Dec"},{"v":1259971200000,"label":"05 Dec"},{"v":1260057600000,"label":"06 Dec"},{"v":1260144000000,"label":"07 Dec"}], ticker(1259625600000, 1260144000000, 480, options));
  assert.deepEqual([{"v":1259625600000,"label":"01 Dec"},{"v":1259712000000,"label":"02 Dec"},{"v":1259798400000,"label":"03 Dec"},{"v":1259884800000,"label":"04 Dec"},{"v":1259971200000,"label":"05 Dec"},{"v":1260057600000,"label":"06 Dec"},{"v":1260144000000,"label":"07 Dec"}], ticker(1259625600000, 1260144000000, 600, options));
  assert.deepEqual([{"v":1260057600000,"label":"06 Dec"},{"v":1260662400000,"label":"13 Dec"},{"v":1261267200000,"label":"20 Dec"},{"v":1261872000000,"label":"27 Dec"},{"v":1262476800000,"label":"03 Jan"},{"v":1263081600000,"label":"10 Jan"},{"v":1263686400000,"label":"17 Jan"},{"v":1264291200000,"label":"24 Jan"}], ticker(1260057600000, 1264291200000, 640, options));
  assert.deepEqual([{"v":1262304000000,"label":"Jan 2010"},{"v":1264982400000,"label":"Feb 2010"},{"v":1267401600000,"label":"Mar 2010"},{"v":1270080000000,"label":"Apr 2010"}], ticker(1262304000000, 1270857600000, 640, options));
  assert.deepEqual([{"v":1288915200000,"label":"05 Nov"},{"v":1288936800000,"label":"06:00"},{"v":1288958400000,"label":"12:00"},{"v":1288980000000,"label":"18:00"},{"v":1289001600000,"label":"06 Nov"},{"v":1289023200000,"label":"06:00"},{"v":1289044800000,"label":"12:00"},{"v":1289066400000,"label":"18:00"},{"v":1289088000000,"label":"07 Nov"},{"v":1289109600000,"label":"06:00"},{"v":1289131200000,"label":"12:00"},{"v":1289152800000,"label":"18:00"},{"v":1289174400000,"label":"08 Nov"},{"v":1289196000000,"label":"06:00"},{"v":1289217600000,"label":"12:00"},{"v":1289239200000,"label":"18:00"},{"v":1289260800000,"label":"09 Nov"}], ticker(1288911600000, 1289260800000, 1024, options));
  assert.deepEqual([{"v":1291161600000,"label":"01 Dec"},{"v":1291248000000,"label":"02 Dec"},{"v":1291334400000,"label":"03 Dec"},{"v":1291420800000,"label":"04 Dec"},{"v":1291507200000,"label":"05 Dec"},{"v":1291593600000,"label":"06 Dec"},{"v":1291680000000,"label":"07 Dec"},{"v":1291766400000,"label":"08 Dec"},{"v":1291852800000,"label":"09 Dec"}], ticker(1291161600000, 1291852800000, 600, options));
  assert.deepEqual([{"v":1294358400000,"label":"07 Jan"},{"v":1294444800000,"label":"08 Jan"},{"v":1294531200000,"label":"09 Jan"},{"v":1294617600000,"label":"10 Jan"},{"v":1294704000000,"label":"11 Jan"},{"v":1294790400000,"label":"12 Jan"},{"v":1294876800000,"label":"13 Jan"},{"v":1294963200000,"label":"14 Jan"}], ticker(1294358400000, 1294963200000, 480, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"}], ticker(1307908000112, 1307908050165, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"}], ticker(1307908000112, 1307908051166, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"}], ticker(1307908000112, 1307908052167, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"}], ticker(1307908000112, 1307908053167, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"}], ticker(1307908000112, 1307908054168, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"}], ticker(1307908000112, 1307908055169, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"}], ticker(1307908000112, 1307908056169, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"}], ticker(1307908000112, 1307908057170, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"}], ticker(1307908000112, 1307908058171, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"}], ticker(1307908000112, 1307908059172, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908060172, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908061174, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908062176, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908063177, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908064178, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908065000,"label":"19:47:45"}], ticker(1307908000112, 1307908065178, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908065000,"label":"19:47:45"}], ticker(1307908000112, 1307908066178, 800, options));
  assert.deepEqual([{"v":1307908005000,"label":"19:46:45"},{"v":1307908010000,"label":"19:46:50"},{"v":1307908015000,"label":"19:46:55"},{"v":1307908020000,"label":"19:47"},{"v":1307908025000,"label":"19:47:05"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908035000,"label":"19:47:15"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908045000,"label":"19:47:25"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908055000,"label":"19:47:35"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908065000,"label":"19:47:45"}], ticker(1307908000112, 1307908067179, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908068179, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"}], ticker(1307908000112, 1307908069179, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908070180, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908071180, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908072181, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908073181, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908074182, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908075182, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908076183, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908077183, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908078184, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"}], ticker(1307908000112, 1307908079185, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908080186, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908081187, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908082188, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908083188, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908084189, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908085190, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908086191, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908087192, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908088192, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"}], ticker(1307908000112, 1307908089193, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908090194, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908091194, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908092196, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908093196, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908094197, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908095197, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908096198, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908097199, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908098200, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"}], ticker(1307908000112, 1307908099200, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908100201, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908101201, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908102202, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908103203, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908104204, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908105205, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908106205, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908107206, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908108209, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"}], ticker(1307908000112, 1307908109209, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908110209, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908111210, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908112211, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908113211, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908114212, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908115213, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908116214, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908117214, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908118215, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908119215, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908120217, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908121218, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908122219, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908123219, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908124220, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908125221, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908126222, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908127222, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908128223, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"}], ticker(1307908000112, 1307908129223, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"},{"v":1307908130000,"label":"19:48:50"}], ticker(1307908000112, 1307908130224, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"},{"v":1307908130000,"label":"19:48:50"}], ticker(1307908000112, 1307908131225, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"},{"v":1307908130000,"label":"19:48:50"}], ticker(1307908000112, 1307908132226, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"},{"v":1307908130000,"label":"19:48:50"}], ticker(1307908000112, 1307908133227, 800, options));
  assert.deepEqual([{"v":1307908010000,"label":"19:46:50"},{"v":1307908020000,"label":"19:47"},{"v":1307908030000,"label":"19:47:10"},{"v":1307908040000,"label":"19:47:20"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908060000,"label":"19:47:40"},{"v":1307908070000,"label":"19:47:50"},{"v":1307908080000,"label":"19:48"},{"v":1307908090000,"label":"19:48:10"},{"v":1307908100000,"label":"19:48:20"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908120000,"label":"19:48:40"},{"v":1307908130000,"label":"19:48:50"}], ticker(1307908000112, 1307908134227, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908135227, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908136228, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908137230, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908138231, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"}], ticker(1307908000112, 1307908139232, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908140233, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908141233, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908142234, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908143240, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908144240, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908145240, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908146241, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908147241, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908148242, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908149243, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908150243, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908151244, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908152245, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908153245, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908154246, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908155247, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908156247, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908157248, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908158249, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908159250, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908160251, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908161252, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908162252, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908163253, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908164254, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908165254, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908166255, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908167256, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908168256, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"}], ticker(1307908000112, 1307908169257, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"},{"v":1307908170000,"label":"19:49:30"}], ticker(1307908000112, 1307908170258, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"},{"v":1307908170000,"label":"19:49:30"}], ticker(1307908000112, 1307908171258, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"},{"v":1307908170000,"label":"19:49:30"}], ticker(1307908000112, 1307908172259, 800, options));
  assert.deepEqual([{"v":1307908020000,"label":"19:47"},{"v":1307908050000,"label":"19:47:30"},{"v":1307908080000,"label":"19:48"},{"v":1307908110000,"label":"19:48:30"},{"v":1307908140000,"label":"19:49"},{"v":1307908170000,"label":"19:49:30"}], ticker(1307908000112, 1307908173260, 800, options));
  assert.deepEqual([{"v":978307200000,"label":"Jan 2001"},{"v":986083200000,"label":"Apr 2001"},{"v":993945600000,"label":"Jul 2001"},{"v":1001894400000,"label":"Oct 2001"}], ticker(978307200000, 1001894400000, 400, options));

  assert.deepEqual([{"v":1307908000110,"label":"40.110"},{"v":1307908000111,"label":"40.111"},{"v":1307908000112,"label":"40.112"},{"v":1307908000113,"label":"40.113"}], ticker(1307908000110, 1307908000113, 200, options));
  assert.deepEqual([{"v":1307908000110,"label":"40.110"},{"v":1307908000112,"label":"40.112"},{"v":1307908000114,"label":"40.114"},{"v":1307908000116,"label":"40.116"}], ticker(1307908000110, 1307908000116, 200, options));
  assert.deepEqual([{"v":1307908000110,"label":"40.110"},{"v":1307908000115,"label":"40.115"},{"v":1307908000120,"label":"40.120"},{"v":1307908000125,"label":"40.125"}], ticker(1307908000110, 1307908000125, 200, options));
  assert.deepEqual([{"v":1307908000110,"label":"40.110"},{"v":1307908000120,"label":"40.120"},{"v":1307908000130,"label":"40.130"},{"v":1307908000140,"label":"40.140"}], ticker(1307908000110, 1307908000140, 200, options));

  assert.deepEqual([{"v":1307908000100,"label":"40.100"},{"v":1307908000150,"label":"40.150"},{"v":1307908000200,"label":"40.200"},{"v":1307908000250,"label":"40.250"}], ticker(1307908000100, 1307908000250, 200, options));
  assert.deepEqual([{"v":1307908000100,"label":"40.100"},{"v":1307908000150,"label":"40.150"},{"v":1307908000200,"label":"40.200"},{"v":1307908000250,"label":"40.250"}], ticker(1307908000090, 1307908000260, 200, options));

  assert.deepEqual([{"v":1307908000100,"label":"40.100"},{"v":1307908000200,"label":"40.200"},{"v":1307908000300,"label":"40.300"},{"v":1307908000400,"label":"40.400"}], ticker(1307908000100, 1307908000400, 200, options));
  assert.deepEqual([{"v":1307908000100,"label":"40.100"},{"v":1307908000200,"label":"40.200"},{"v":1307908000300,"label":"40.300"},{"v":1307908000400,"label":"40.400"}], ticker(1307908000100, 1307908000410, 200, options));

  assert.deepEqual([{"v":1307908000000,"label":"40.000"},{"v":1307908000500,"label":"40.500"},{"v":1307908001000,"label":"41.000"},{"v":1307908001500,"label":"41.500"}], ticker(1307908000000, 1307908001500, 200, options));
});

});
