# Authors:
#     Ade Lee <alee@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Copyright (C) 2013 Red Hat, Inc.
# All rights reserved.
#

from __future__ import absolute_import
import os
from lxml import etree as ET

import pki
import pki.server.upgrade


class CloningInterfaceChanges(pki.server.upgrade.PKIServerUpgradeScriptlet):

    updateDomainServletData = """
        <servlet>
             <servlet-name>  caUpdateDomainXML-admin  </servlet-name>
             <servlet-class> com.netscape.cms.servlet.csadmin.UpdateDomainXML  </servlet-class>
             <init-param>
                 <param-name>  GetClientCert  </param-name>
                 <param-value> false          </param-value>
             </init-param>
             <init-param>
                 <param-name>  authority   </param-name>
                 <param-value> ca          </param-value>
             </init-param>
             <init-param>
                 <param-name>  ID          </param-name>
                 <param-value> caUpdateDomainXML </param-value>
             </init-param>
             <init-param>
                 <param-name>  interface   </param-name>
                 <param-value> admin       </param-value>
             </init-param>
             <init-param>
                 <param-name>  AuthMgr     </param-name>
                 <param-value> TokenAuth </param-value>
             </init-param>
             <init-param>
                 <param-name>  AuthzMgr    </param-name>
                 <param-value> BasicAclAuthz </param-value>
             </init-param>
             <init-param>
                 <param-name>  resourceID  </param-name>
                 <param-value> certServer.securitydomain.domainxml </param-value>
             </init-param>
         </servlet> """

    updateDomainMappingData = """
        <servlet-mapping>
            <servlet-name>  caUpdateDomainXML-admin </servlet-name>
            <url-pattern>   /admin/ca/updateDomainXML  </url-pattern>
        </servlet-mapping> """

    tokenAuthenticateServletData = """
        <servlet>
            <servlet-name>  caTokenAuthenticate-admin  </servlet-name>
            <servlet-class> com.netscape.cms.servlet.csadmin.TokenAuthenticate  </servlet-class>
            <init-param>
                 <param-name>  GetClientCert  </param-name>
                 <param-value> false       </param-value>
            </init-param>
            <init-param>
                 <param-name>  authority   </param-name>
                 <param-value> ca          </param-value>
            </init-param>
            <init-param>
                 <param-name>  ID          </param-name>
                 <param-value> caTokenAuthenticate  </param-value>
            </init-param>
            <init-param>
                <param-name>  interface   </param-name>
                <param-value> admin       </param-value>
            </init-param>
        </servlet>"""

    tokenAuthenticateMappingData = """
        <servlet-mapping>
            <servlet-name>  caTokenAuthenticate-admin </servlet-name>
            <url-pattern>   /admin/ca/tokenAuthenticate  </url-pattern>
        </servlet-mapping>"""

    def __init__(self):
        super(CloningInterfaceChanges, self).__init__()
        self.message = 'Change interfaces for cloning'
        self.doc = None
        self.root = None

    def upgrade_subsystem(self, instance, subsystem):
        web_xml = os.path.join(
            instance.base_dir,
            'webapps', subsystem.name,
            'WEB-INF', 'web.xml')

        self.backup(web_xml)

        self.doc = ET.parse(web_xml)
        self.root = self.doc.getroot()
        self.remove_get_token_info(subsystem.name)
        if subsystem.name == "ca":
            self.modify_update_number_range(subsystem.name)
            self.modify_update_domain_xml()
            self.modify_token_authenticate()
        if subsystem.name == "kra":
            self.modify_update_number_range(subsystem.name)

        self.doc.write(web_xml)

    def modify_update_number_range(self, subsystem):
        # change servlet and mapping for caUpdateNumberRange
        for servlet in self.doc.findall('.//servlet'):
            name = servlet.find('servlet-name').text.strip()
            if name == (subsystem + "UpdateNumberRange"):
                for param in servlet.findall('.//init-param'):
                    pname = param.find('param-name').text.strip()
                    if pname == 'interface':
                        pvalue = param.find('param-value')
                        pvalue.text = 'admin'

        for mapping in self.doc.findall('.//servlet-mapping'):
            name = mapping.find('servlet-name').text.strip()
            if name == (subsystem + "UpdateNumberRange"):
                url_pattern = mapping.find('url-pattern')
                url_pattern.text = '/admin/' + subsystem + '/updateNumberRange'

    def remove_get_token_info(self, subsystem):
        # remove getTokenInfo servlet and servlet mapping
        for servlet in self.doc.findall('.//servlet'):
            name = servlet.find('servlet-name').text.strip()
            if name == (subsystem + "GetTokenInfo"):
                self.root.remove(servlet)

        for mapping in self.doc.findall('.//servlet-mapping'):
            name = mapping.find('servlet-name').text.strip()
            if name == (subsystem + "GetTokenInfo"):
                self.root.remove(mapping)

    def modify_update_domain_xml(self):
        # add caUpdateDomainXML-admin servlet and mapping
        found = False
        for servlet in self.doc.findall('.//servlet'):
            name = servlet.find('servlet-name').text.strip()
            if name == 'caUpdateDomainXML-admin':
                found = True
            if name == 'caUpdateDomainXML':
                index = self.root.index(servlet) + 1
        if not found:
            servlet = ET.fromstring(self.updateDomainServletData)
            self.root.insert(index, servlet)

        found = False
        for mapping in self.doc.findall('.//servlet-mapping'):
            name = mapping.find('servlet-name').text.strip()
            if name == 'caUpdateDomainXML-admin':
                found = True
            if name == 'caUpdateDomainXML':
                index = self.root.index(mapping) + 1
        if not found:
            mapping = ET.fromstring(self.updateDomainMappingData)
            self.root.insert(index, mapping)

    def modify_token_authenticate(self):
        # add caTokenAuthenticate-admin servlet and mapping
        found = False
        for servlet in self.doc.findall('.//servlet'):
            name = servlet.find('servlet-name').text.strip()
            if name == 'caTokenAuthenticate-admin':
                found = True
            if name == 'caTokenAuthenticate':
                index = self.root.index(servlet) + 1
        if not found:
            servlet = ET.fromstring(self.tokenAuthenticateServletData)
            self.root.insert(index, servlet)

        found = False
        for mapping in self.doc.findall('.//servlet-mapping'):
            name = mapping.find('servlet-name').text.strip()
            if name == 'caTokenAuthenticate-admin':
                found = True
            if name == 'caTokenAuthenticate':
                index = self.root.index(mapping) + 1
        if not found:
            mapping = ET.fromstring(self.tokenAuthenticateMappingData)
            self.root.insert(index, mapping)
