//
// C++ Implementation: BoatGraphicsItem
//
// Description:
//
//
// Author: Thibaut GRIDEL <tgridel@free.fr>
//
// Copyright (c) 2008-2015 Thibaut GRIDEL
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//
#include "boat.h"

#include "commontypes.h"
#include "situationscene.h"
#include "situationmodel.h"
#include "trackmodel.h"
#include "boatmodel.h"

#include "boatsengine.h"
#include "boatproperties.h"

#include <QPainter>
#include <QGraphicsSceneMouseEvent>
#include <QGraphicsScene>

#include <iostream>

extern int debugLevel;

BoatGraphicsItem::BoatGraphicsItem(BoatModel *boat, QGraphicsItem *parent)
        : QGraphicsItem(parent),
        m_boat(boat),
        m_angle(0),
        m_sail(new SailGraphicsItem(boat, this)),
        m_jib(new SailGraphicsItem(boat, this)),
        m_spin(new SpinnakerGraphicsItem(boat, this)),
        m_genn(new GennakerGraphicsItem(boat, this)),
        m_hasSpin(false),
        m_hasGenn(false),
        m_overlap(Boats::none),
        m_overlapLine(new QGraphicsLineItem(this)),
        m_color(boat->track()->color()),
        m_flag(Boats::noFlag),
        m_flagRect(new FlagGraphicsItem(this)),
        m_hidden(false),
        m_bubble(new BubbleGraphicsItem(m_boat, this)),
        m_series(-1),
        m_order(0),
        m_numberPath(new QGraphicsPathItem(this)),
        m_laylines(new LaylinesGraphicsItem(boat, this)) {
    setFlag(QGraphicsItem::ItemIsMovable);
    setFlag(QGraphicsItem::ItemIsSelectable);

    setBoundingRegionGranularity(1);

    m_numberPath->setZValue(1);
    m_flagRect->setZValue(2);
    m_sail->setZValue(3);
    m_jib->setZValue(4);
    m_spin->setZValue(5);
    m_genn->setZValue(5); // Make this same as for spinnaker on basis that a boat will have one or other of these, not both
    m_bubble->setZValue(6);

    m_numberPath->setBrush(QBrush(Qt::black));

    QPen dashPen(Qt::CustomDashLine);
    QVector<qreal> dashes;
    dashes << 5 << 5;
    dashPen.setDashPattern(dashes);
    m_overlapLine->setPen(dashPen);

    setSeries(boat->track()->series());
    setHeading(boat->heading());
    setPos(boat->position());
    setOrder(boat->order());
    setDim(boat->dim());
    m_sail->setSailAngle(m_boat->trimmedSailAngle());
    m_jib->setSailAngle(m_boat->trimmedJibAngle());
    m_spin->setHeading(m_boat->heading());
    m_spin->setSailAngle(m_boat->trimmedSpinAngle());
    m_genn->setSailAngle(m_boat->trimmedGennAngle()); // Use spinTrim() for both spinnaker and gennaker
    setSpin(boat->spin());
    setOverlap(boat->overlap());
    setDisplayFlag(boat->flag());
    setHidden(boat->hidden());

    connect(boat, SIGNAL(headingChanged(qreal)),
            this, SLOT(setHeading(qreal)));
    connect(boat, SIGNAL(headingChanged(qreal)),
            m_spin, SLOT(setHeading(qreal)));
    connect(boat, SIGNAL(positionChanged(QPointF)),
            this, SLOT(setPosition(QPointF)));
    connect(boat, SIGNAL(trimmedSailAngleChanged(qreal)),
            m_sail, SLOT(setSailAngle(qreal)));
    connect(boat, SIGNAL(trimmedJibAngleChanged(qreal)),
            m_jib, SLOT(setSailAngle(qreal)));
    connect(boat, SIGNAL(spinChanged(bool)),
            this, SLOT(setSpin(bool)));
    connect(boat, SIGNAL(trimmedSpinAngleChanged(qreal)),
            m_spin, SLOT(setSailAngle(qreal)));
    connect(boat, SIGNAL(trimmedGennAngleChanged(qreal)),
            m_genn, SLOT(setSailAngle(qreal)));
    connect(boat, SIGNAL(orderChanged(int)),
            this, SLOT(setOrder(int)));
    connect(boat, SIGNAL(overlapChanged(Boats::Overlaps)),
            this, SLOT(setOverlap(Boats::Overlaps)));
    connect(boat, SIGNAL(flagChanged(Boats::Flag)),
            this, SLOT(setDisplayFlag(Boats::Flag)));
    connect(boat, SIGNAL(hiddenChanged(bool)),
            this, SLOT(setHidden(bool)));
    connect(boat, SIGNAL(dimChanged(int)),
            this, SLOT(setDim(int)));
    connect(boat->track(), SIGNAL(colorChanged(QColor)),
            this, SLOT(setColor(QColor)));
    connect(boat->track(), SIGNAL(seriesChanged(int)),
            this, SLOT(setSeries(int)));
    connect(boat->situation(), SIGNAL(boatRemoved(BoatModel*)),
            this, SLOT(deleteItem(BoatModel*)));
    connect(boat->track(), SIGNAL(trackSelected(bool)),
            this, SLOT(setSelected(bool)));
}


BoatGraphicsItem::~BoatGraphicsItem() {}

void BoatGraphicsItem::setHeading(qreal value) {
    if (m_angle != value) {
        m_angle = value;
        QTransform rotation;
        rotation.rotate(m_angle),
        setTransform(rotation, false);
    }
}

void BoatGraphicsItem::setPosition(QPointF position) {
    if (pos() != position) {
        setPos(position);
        update();
    }
}

void BoatGraphicsItem::setSpin(bool value) {
    m_spin->setVisible(m_hasSpin && value);
    m_genn->setVisible(m_hasGenn && value);
}

void BoatGraphicsItem::setOrder(int value) {
    m_order = value;
    if (m_order && m_numberSize) {
        QString number = QString::number(m_order);
        QFont numberFont;
        numberFont.setPointSize(m_numberSize);
        QFontMetrics fm(numberFont);
        QPainterPath fpath;
        fpath.addText(-fm.width(number)/2.0, 0, numberFont, number);
        m_numberPath->setPath(fpath);
        setZValue(m_order);
    } else {
        m_numberPath->setPath(QPainterPath());
        setZValue(boat()->track()->size()+1);
    }

    update();
}

void BoatGraphicsItem::setOverlap(Boats::Overlaps value) {
    if (m_overlap != value) {
        m_overlap = value;
        setOverlapLine();
    }
}

void BoatGraphicsItem::setOverlapLine() {
    qreal size = m_boat->situation()->engine()->seriesProperties(m_series)->size();
    QLineF line;
    line.setP1(QPointF(m_border, size/2));
    line.setP2(QPointF(-m_border, size/2));
    if (m_overlap == Boats::none) {
        m_overlapLine->setVisible(false);
    } else {
        if (m_overlap & Boats::starboard) {
            line.setP2(QPointF(m_border + size, size/2));
        }
        if (m_overlap & Boats::port) {
            line.setP1(QPointF(-m_border - size, size/2));
        }
        m_overlapLine->setVisible(!m_hidden && true);
        m_overlapLine->setLine(line);
    }
}

void BoatGraphicsItem::setDisplayFlag(Boats::Flag value) {
    m_flag = value;
    m_flagRect->setDisplayFlag(m_flag);

    if (m_flag == Boats::noFlag) {
        m_flagRect->setVisible(false);
    } else {
        m_flagRect->setVisible(!m_hidden && true);
    }
}

void BoatGraphicsItem::setHidden(bool value) {
    prepareGeometryChange();
    m_hidden = value;
    if (m_hidden) {
        m_sail->setVisible(false);
        m_jib->setVisible(false);
        m_spin->setVisible(false);
        m_genn->setVisible(false);
        m_overlapLine->setVisible(false);
        m_flagRect->setVisible(false);
        m_bubble->setVisible(false);
        m_numberPath->setVisible(false);
        m_laylines->setVisible(false);
    } else {
        m_sail->setVisible(true);
        m_jib->setVisible(true);
        m_spin->setVisible(boat()->spin());
        m_genn->setVisible(boat()->spin());
        m_overlapLine->setVisible(m_overlap != Boats::none);
        m_flagRect->setVisible(m_flag != Boats::noFlag);
        m_bubble->setVisible(!boat()->text().isEmpty());
        m_numberPath->setVisible(true);
        m_laylines->setVisible(boat()->laylines());
    }

}

void BoatGraphicsItem::setColor(QColor value) {
    if (m_color != value) {
        int alpha = m_color.alpha();
        m_color = value;
        m_color.setAlpha(alpha);
        update();
    }
}

void BoatGraphicsItem::setDim(int value) {
    m_color.setAlpha(value);
    if (value == 0 ) {
        setVisible(false);
    }
    else {
        setVisible(true);
        if (value != 255) {
            int maxSize = 0;
            foreach (const TrackModel *track, boat()->situation()->tracks()) {
                if (track->boats().size() > maxSize)
                    maxSize = track->boats().size() - 1;
            }
            setZValue(maxSize+1);
        } else {
            setZValue(boat()->order());
        }
    }
    update();
}

void BoatGraphicsItem::setVisible(bool value) {
    QGraphicsItem::setVisible(value);
}

void BoatGraphicsItem::setSeries(int value) {
    if (m_series != value) {
        prepareGeometryChange();
        m_series = value;
        BoatProperties* properties = m_boat->situation()->engine()->seriesProperties(m_series);

        m_hullPath = properties->path();
        m_boundingRect = m_hullPath.boundingRect();
        m_border = properties->border();

        m_numberSize = properties->numberSize();
        m_numberPath->setPos(properties->numberPos());

        m_flagRect->setRect(properties->flagRect());

        if (properties->hasSail()) {
            m_sail->setPosition(properties->mastPos());
            m_sail->setSailSize(properties->sailSize());
            m_sail->setVisible(true);
        } else {
            m_sail->setVisible(false);
        }

        if (properties->hasJib()) {
            m_jib->setPosition(properties->jibTackPos());
            m_jib->setSailSize(properties->jibSize());
            m_jib->setVisible(true);
        } else {
            m_jib->setVisible(false);
        }

        m_hasSpin = properties->hasSpin();
        m_hasGenn = properties->hasGennaker();
        // set setHasSpin true irrespective of type of spinnaker
        // (is just used to control whether Toggle Spinnaker menu item is active)

        if (m_hasSpin) {
            m_spin->setPosition(properties->mastPos());
            m_spin->setSailSize(properties->spinSize());
            m_spin->setVisible(m_boat->spin());
        } else {
            m_spin->setVisible(false);
        }

        if (m_hasGenn) {
            m_genn->setPosition(properties->gennakerTackPos());
            m_genn->setPoleLength(properties->gennakerPoleSize());
            m_genn->setSailSize(properties->gennakerSize());
            m_genn->setVisible(m_boat->spin());
        } else {
            m_genn->setVisible(false);
        }

        setOverlapLine();
        setOrder(m_order);
        update();
    }
}

void BoatGraphicsItem::deleteItem(BoatModel *boat) {
    if (boat == m_boat) {
        if (debugLevel & 1 << VIEW) std::cout << "deleting boatGraphics for model" << m_boat << std::endl;
        scene()->removeItem(this);
        delete this;
    }
}

void BoatGraphicsItem::mousePressEvent(QGraphicsSceneMouseEvent *event) {
    m_multiSelect = (event->modifiers() & Qt::ControlModifier) != 0;
    m_trackSelect = (event->modifiers() & Qt::ShiftModifier) != 0;

    bool selection = true;
    if (m_multiSelect) {
        selection = !isSelected();
    }

    if (m_trackSelect) {
        m_boat->track()->setSelected(selection);
    } else {
        if (selection) {
            m_boat->situation()->addSelectedBoat(m_boat);
        } else {
            m_boat->situation()->removeSelectedModel(m_boat);
        }
        setSelected(selection);
    }

    update();
}

void BoatGraphicsItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event) {
    Q_UNUSED(event);
}

void BoatGraphicsItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event) {
    Q_UNUSED(event);
}

void BoatGraphicsItem::setSelected(bool selected) {
    QGraphicsItem::setSelected(selected);
}

QRectF BoatGraphicsItem::boundingRect() const {
    if (m_hidden)
        return QRectF(-3, -3, 6, 6);
    else
        return m_boundingRect;
}

QPainterPath BoatGraphicsItem::shape() const {
    QPainterPath path;
    path.addRegion(boundingRegion(QTransform()));
    return path;
}

void BoatGraphicsItem::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,

                             QWidget *widget) {
    Q_UNUSED(option);
    Q_UNUSED(widget);

    if (isSelected())
        painter->setPen(Qt::red);
    else
        painter->setPen(Qt::black);
    painter->setBrush(m_color);
    if (m_hidden)
        painter->drawEllipse(-3, -3, 6, 6);
    else
        painter->drawPath(m_hullPath);

}

int BoatGraphicsItem::type() const {
    return BOAT_TYPE;
}
