/*################################################################################
# Linux Management Providers (LMP), Simple Identity Management provider package
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Khahramon NURIDDINOV, TUIT <qahramon0786@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
################################################################################*/

#include "OpenDRIM_AccountAccess.h"

#include <sys/stat.h>

string SystemName;
vector<string> GID;
vector<string> GNames;

int Simple_Identity_Management_OpenDRIM_Account_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getSystemName(SystemName, errorMessage));
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_Account>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	// Read the /etc/passwd file
	string etc_passwd;
	
	CF_assert(CF_readTextFile("/etc/passwd", etc_passwd, errorMessage));
	
	vector<string> etc_passwd_lines;
	CF_splitText(etc_passwd_lines, etc_passwd, '\n');
	// Read the /etc/shadow file
	string etc_shadow;
	vector<string> etc_shadow_lines;
	if (discriminant=="ei" || discriminant=="gi") {
	
		CF_assert(CF_readTextFile("/etc/shadow", etc_shadow, errorMessage));
		
		CF_splitText(etc_shadow_lines, etc_shadow, '\n');
		if (etc_shadow_lines.size()!=etc_passwd_lines.size()) {
			errorMessage="Wrong format (missmatch with /etc/passwd): /etc/shadow";
			return FAILED;
		}
	}
	// Get the lists of the groups
	vector<OpenDRIM_Group> groups;
	CF_assert(Simple_Identity_Management_OpenDRIM_Group_retrieve(broker, ctx, groups, NULL, errorMessage, "ei"));
	// In gi (getInstance) mode the first instance in the vector contains
	// the objectpath of the instance we're supposed to get
	string gi_name;
	if (discriminant=="gi")
		result[0].getName(gi_name);
	// For each line of /etc/passwd and /etc/shadow
	for (size_t i=0; i<etc_passwd_lines.size(); i++) {
		// Check the format of the line in /etc/passwd
		if (etc_passwd_lines[i]=="")
			break;
		vector<string> current_etc_passwd_line;
		CF_splitText(current_etc_passwd_line, etc_passwd_lines[i], ':');
		if (current_etc_passwd_line.size()!=7) {
			errorMessage="Wrong format (at line "+CF_intToStr((unsigned long) i+1)+"): /etc/passwd";
			return FAILED;
		}
		// Assign the key values
		OpenDRIM_Account instance;
		instance.setSystemName(SystemName);
		instance.setName(current_etc_passwd_line[0]);
		instance.setCreationClassName("OpenDRIM_Account");
		instance.setSystemCreationClassName("OpenDRIM_ComputerSystem");
		
		// Set OrganizationName
		vector<string> organizationNames;
		organizationNames.push_back("unknown");
		if(discriminant=="ei")
			instance.setOrganizationName(organizationNames);
		else if(discriminant=="gi")
			result[0].setOrganizationName(organizationNames);
		
		// If we need the content of the instance(s)
		// We check if the current line matches with the requested instance (gi mode)
		if (discriminant=="ei" || (discriminant=="gi" && CF_strCmpNoCase(gi_name, current_etc_passwd_line[0]))) {
			// Check the format of the line of /etc/shadow
			vector<string> current_etc_shadow_line;
			CF_splitText(current_etc_shadow_line, etc_shadow_lines[i], ':');
			if (current_etc_shadow_line.size()!=9) {
				errorMessage="Wrong format (at line "+CF_intToStr((unsigned long) i+1)+"): /etc/shadow";
				return FAILED;
			}
			// If we're in gi mode, fill in the first instance of the vector
			if (discriminant=="gi") {
				CF_assert(Simple_Identity_Management_OpenDRIM_Account_extractAccountInfo(result[0], current_etc_passwd_line, current_etc_shadow_line, groups, errorMessage));
				// We found the instance so we can exit here
				return OK;
			}
			// Else we fill the current instance
			CF_assert(Simple_Identity_Management_OpenDRIM_Account_extractAccountInfo(instance, current_etc_passwd_line, current_etc_shadow_line, groups, errorMessage));
		}
		// Add the instance to the vector
		if (discriminant=="ei" || discriminant=="ein")
			result.push_back(instance);
	}
	// If we get here it means we didn't find the instance
	if (discriminant=="gi")
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_Account& instance, const char** properties, string& errorMessage) {
	_E_;
	// Check the key values
	string _SystemName;
	string CreationClassName;
	string SystemCreationClassName;
	instance.getCreationClassName(CreationClassName);
	instance.getSystemCreationClassName(SystemCreationClassName);
	instance.getSystemName(_SystemName);
	
	if (!CF_strCmpNoCase(_SystemName, SystemName) || !CF_strCmpNoCase(CreationClassName, "OpenDRIM_Account") || !CF_strCmpNoCase(SystemCreationClassName, "OpenDRIM_ComputerSystem"))
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}
	
	// Get the instance (it also checks if it exists)
	vector<OpenDRIM_Account> instances;
	instances.push_back(instance);
	
	CF_assert(Simple_Identity_Management_OpenDRIM_Account_retrieve(broker, ctx, instances, properties, errorMessage, "gi"));
	
	instance=instances[0];
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& newInstance, const OpenDRIM_Account& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// We are going to modify the instance by executing the 'usermod' command
	string usermod_command="/usr/sbin/usermod ";
	string newElementName, oldElementName;
	// Check which properties have been modified and form usermod options
	if (newInstance.getElementName(newElementName)!=NOT_FOUND) {
		if (oldInstance.getElementName(oldElementName)==NOT_FOUND || newElementName!=oldElementName)
			usermod_command+="-c \""+newElementName+"\" ";
	}
	string newPrimaryGroup, oldPrimaryGroup;
	if (newInstance.getPrimaryGroup(newPrimaryGroup)!=NOT_FOUND) {
		if (oldInstance.getPrimaryGroup(oldPrimaryGroup)==NOT_FOUND || newPrimaryGroup!=oldPrimaryGroup)
			usermod_command+="-g "+newPrimaryGroup+" ";
	}
	string newLoginShell, oldLoginShell;
	if (newInstance.getLoginShell(newLoginShell)!=NOT_FOUND) {
		if (oldInstance.getLoginShell(oldLoginShell)==NOT_FOUND || newLoginShell!=oldLoginShell)
			usermod_command+="-s "+newLoginShell+" ";
	}
	string newHomeDirectory, oldHomeDirectory;
	if (newInstance.getHomeDirectory(newHomeDirectory)!=NOT_FOUND) {
		if (oldInstance.getHomeDirectory(oldHomeDirectory)==NOT_FOUND || newHomeDirectory!=oldHomeDirectory)
			usermod_command+="-d "+newHomeDirectory+" -m ";
	}
	string newAccountExpiration, oldAccountExpiration;
	newInstance.getAccountExpiration(newAccountExpiration);
	if (newInstance.getAccountExpiration(newAccountExpiration)!=NOT_FOUND) {
		if (oldInstance.getAccountExpiration(oldAccountExpiration)==NOT_FOUND || newAccountExpiration!=oldAccountExpiration)
			usermod_command+="-e "+newAccountExpiration.substr(0,4)+"-"+newAccountExpiration.substr(4,2)+"-"+newAccountExpiration.substr(6,2)+" ";
	}
	// Get the default minimum length for the password
	vector<string> newUserPassword;
	vector<string> etc_login_defs_entries, etc_login_defs_values;
	etc_login_defs_entries.push_back("PASS_MIN_LEN");
	
	CF_assert(Simple_Identity_Management_OpenDRIM_Account_extractLoginDefsValue(etc_login_defs_values, etc_login_defs_entries, errorMessage));
	
	if (newInstance.getUserPassword(newUserPassword)!=NOT_FOUND && newUserPassword.size()>0) {
		// MD5 encryption
		string salt=Simple_Identity_Management_OpenDRIM_Account_generateMD5Salt();
		string password;
		for (size_t i=0; i<newUserPassword.size(); i++)
			password+=newUserPassword[i];
		if (etc_login_defs_values[0]!="" && password.size()<(unsigned int)atol(etc_login_defs_values[0].c_str())) {
			errorMessage+="Account passwords MUST be a least "+etc_login_defs_values[0]+" character(s) long.";
			return FAILED;
		}
		usermod_command+=" -p '"+(string) crypt(password.c_str(), salt.c_str())+"' ";
	}
	// Execute usermod
	if (usermod_command!="/usr/sbin/usermod ") {
		string Name;
		oldInstance.getName(Name);
		string stdOut, stdErr;
		
		CF_assert(CF_runCommand(usermod_command+Name, stdOut, stdErr, errorMessage));
	}
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int Simple_Identity_Management_OpenDRIM_Account_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& instance, string& errorMessage) {
	_E_;
	// Simply invoke userdel
	string Name;
	instance.getName(Name);
	string stdOut, stdErr;
	
	CF_assert(CF_runCommand("/usr/sbin/userdel "+Name, stdOut, stdErr, errorMessage));
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& instance, unsigned int& returnValue, const OpenDRIM_Account_RequestStateChange_In& in, OpenDRIM_Account_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// Check the parameters
	string TimeoutPeriod;
	if (in.getTimeoutPeriod(TimeoutPeriod)!=NOT_FOUND) {
		returnValue=4098;
		return OK;
	}
	unsigned short EnabledState=0;
	instance.getEnabledState(EnabledState);
	unsigned short RequestedState;
	if (in.getRequestedState(RequestedState)==NOT_FOUND) {
		returnValue=5;
		return OK;
	}
	if (RequestedState!=2 && RequestedState!=32768) {
		returnValue=5;
		return OK;
	}
	string Name;
	instance.getName(Name);
	string command;
	// Check for an invalid transition and form the usermod command
	if (EnabledState==2 && RequestedState==32768)
		command="/usr/sbin/usermod -L "+Name;
	else if (EnabledState==11 && RequestedState==2)
		command="/usr/sbin/usermod -U "+Name;
	else {
		returnValue=4097;
		return OK;
	}
	// Run usermod
	string stdErr, stdOut;
	int errorCode=CF_runCommand(command, stdOut, stdErr, errorMessage);
	if (errorCode!=OK) {
		returnValue=4;
		return OK;
	}
	returnValue=0;
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_CreateHomeDirectory(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	string HomeDirectory;
	instance.getHomeDirectory(HomeDirectory);
	string Name;
	instance.getName(Name);
	string PrimaryGroup;
	instance.getPrimaryGroup(PrimaryGroup);
	// First we check is the directory exists
	string stdOut, stdErr;
	int errorCode=CF_runCommand("test -d "+HomeDirectory, stdOut, stdErr, errorMessage);
	if (errorCode==OK) {
		returnValue=1;
		return OK;
	}
	// Get the default access mask for home directories
	vector<string> etc_login_defs_entries, etc_login_defs_values;
	etc_login_defs_entries.push_back("UMASK");
	
	CF_assert(Simple_Identity_Management_OpenDRIM_Account_extractLoginDefsValue(etc_login_defs_values, etc_login_defs_entries, errorMessage));
	
	// Security: no directory manipulation outside of '/home/'
	if (HomeDirectory.find("/home/")!=string::npos && HomeDirectory.find("..")==string::npos && HomeDirectory.size()>6) {
		string mode = "700";
		if (etc_login_defs_values.size()==1)
			mode=CF_intToStr(777 - atoi(etc_login_defs_values[0].c_str()));
		// Copy the skeleton (default bash settings, etc.) from /etc/skel
		errorCode=CF_runCommand("cp -rf /etc/skel/ "+HomeDirectory+"/", stdOut, stdErr, errorMessage);
		if (errorCode!=OK) {
			DEBUG;
			returnValue=2;
			return OK;
		}
		// Change the permissions
		errorCode=CF_runCommand("chmod " + mode + " " + HomeDirectory, stdOut, stdErr, errorMessage);
		if (errorCode!=OK) {
			DEBUG;
			returnValue=2;
			return OK;
		}
		// Change the owner to the owner of the account
		errorCode=CF_runCommand("chown -R "+Name+" "+HomeDirectory, stdOut, stdErr, errorMessage);
		if (errorCode!=OK) {
			DEBUG;
			returnValue=2;
			return OK;
		}
		// Chance the group to the primary group of the account
		errorCode=CF_runCommand("chgrp -R "+PrimaryGroup+" "+HomeDirectory, stdOut, stdErr, errorMessage);
		if (errorCode!=OK) {
			DEBUG;
			returnValue=2;
			return OK;
		}
	} else {
		DEBUG;
		returnValue=2;
		return OK;
	}
	returnValue=0;
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_DeleteHomeDirectory(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_Account& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	string HomeDirectory;
	instance.getHomeDirectory(HomeDirectory);
	string stdOut, stdErr;
	// Check if the directory exists
	int errorCode=CF_runCommand("test -d "+HomeDirectory, stdOut, stdErr, errorMessage);
	if (errorCode!=OK) {
		returnValue=1;
		return OK;
	}
	// If we're in '/home/'we can delete
	if (HomeDirectory.find("/home/")!=string::npos && HomeDirectory.find("..")==string::npos && HomeDirectory.size()>6) {
		errorCode=CF_runCommand("rm -r -f "+HomeDirectory, stdOut, stdErr, errorMessage);
		if (errorCode!=OK) {
			returnValue=2;
			return OK;
		}
	}
	else {
		returnValue=2;
		return OK;
	}
	returnValue=0;
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_Account_extractAccountInfo(OpenDRIM_Account& instance, const vector<string>& current_etc_passwd_line, const vector<string>& current_etc_shadow_line, const vector<OpenDRIM_Group>& groups, string& errorMessage) {
	_E_;
	// Assign account information
	instance.setUserID(current_etc_passwd_line[2]);
	instance.setHomeDirectory(current_etc_passwd_line[5]);
	instance.setLoginShell(current_etc_passwd_line[6]);
	// An account is disabled if the account expiration is set
	// or if the password has expired and the limit period
	// to change is passed.
	// The password is locked if it's marked as locked (invalid charater)
	// or if the password expiration date is reached
	// 2: Enabled
	// 3: Disabled
	// 11: Password locked
	unsigned short account_state=2;
	if (current_etc_shadow_line[1].size()==0 || current_etc_shadow_line[1].find('!')!=string::npos || current_etc_shadow_line[1].find('*')!=string::npos) {
		vector<string> UserPassword;
		UserPassword.push_back("!");
		instance.setUserPassword(UserPassword);
		account_state=11;
	}
	time_t current_time=CF_localTime();
	time_t _PasswordLastChange=atol(current_etc_shadow_line[2].c_str())*24*3600;
	string PasswordLastChange=CF_timeToString(_PasswordLastChange);
	instance.setPasswordLastChange(PasswordLastChange);
	if (current_etc_shadow_line[7]!="") {
		time_t _AccountExpiration=atol(current_etc_shadow_line[7].c_str())*24*3600;
		string AccountExpiration=CF_timeToString(_AccountExpiration);
		instance.setAccountExpiration(AccountExpiration);
		if (current_time>_AccountExpiration)
			account_state=3;
	}
	if (current_etc_shadow_line[4]!="" && current_etc_shadow_line[4]!="99999") {
		time_t _PasswordExpiration=atol(current_etc_shadow_line[2].c_str())*24*3600+atol(current_etc_shadow_line[4].c_str())*24*3600;
		string PasswordExpiration=CF_timeToString(_PasswordExpiration);
		instance.setPasswordExpiration(PasswordExpiration);
		if (account_state!=3 && current_time>_PasswordExpiration)
			account_state=11;
		if (current_etc_shadow_line[6]!="") {
			time_t _AccountInactive=_PasswordExpiration+atol(current_etc_shadow_line[6].c_str())*24*3600;
			if (current_time>_AccountInactive)
				account_state=3;
		}
	}
	if (current_etc_passwd_line[4]!="")
		instance.setElementName(current_etc_passwd_line[4]);
	instance.setEnabledState(account_state);
	if (account_state==11)
		account_state=32768;
	instance.setRequestedState(account_state);
	string PrimaryGroup, stdErr;
	// More efficient
	for (vector<OpenDRIM_Group>::size_type i = 0; i < groups.size(); i++) {
		if (current_etc_passwd_line[3] == groups[i].GroupID) {
			PrimaryGroup = groups[i].Name;
			break;
		}
	}
	// Get the GroupID of the primary group
	//CF_assert(CF_runCommand("cat /etc/group | grep "+current_etc_passwd_line[3], PrimaryGroup, stdErr, errorMessage));
	//PrimaryGroup=PrimaryGroup.substr(0, PrimaryGroup.find(':'));
	instance.setPrimaryGroup(PrimaryGroup);
	_L_;
	return OK;
}

// Extract information from /etc/login.defs
int Simple_Identity_Management_OpenDRIM_Account_extractLoginDefsValue(vector<string>& values, const vector<string>& entries, string& errorMessage) {
	_E_;
	values.clear();
	for (size_t j=0; j<entries.size(); j++)
		values.push_back("");
	string login_defs;
	// /etc/login.defs contains the default settings for user accounts
	
	CF_assert(CF_readTextFile("/etc/login.defs", login_defs, errorMessage));
	
	vector<string> login_defs_lines;
	CF_splitText(login_defs_lines, login_defs, '\n');
	for (size_t i=0; i<login_defs_lines.size(); i++) {
		string login_defs_line=login_defs_lines[i].substr(0, login_defs_lines[i].find('#'));
		// Replace tabs by spaces
		for (size_t j=0; j<login_defs_line.size(); j++) {
			if (login_defs_line[j] == '\t')
				login_defs_line[j] = ' ';
		}
		vector<string> login_defs_line_elements;
		CF_splitTextBySpace(login_defs_line_elements, login_defs_line);
		// If the line contains less than 2 elements, skip and go to the next line
		if (login_defs_line_elements.size()<2)
			continue;
		for (size_t j=0; j<entries.size(); j++) {
			if (entries[j]==login_defs_line_elements[0])
				values[j]=login_defs_line_elements[1];
		}
	}
	_L_;
	return OK;
}

// Generate a random salt for MD5 encryption
string Simple_Identity_Management_OpenDRIM_Account_generateMD5Salt () {
	_E_;
	string _salt;
	while(_salt.size()<8) {
		char seed=rand()%123;
		if (seed>=46 && seed<=57 || seed>=65 && seed<=90 || seed>=97)
			_salt.push_back(seed);
	}
	_L_;
	return _salt="$1$" + _salt + "$";
}

int Simple_Identity_Management_OpenDRIM_Account_populate(OpenDRIM_Account& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

