//  (C) Copyright John Maddock 2006-7.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)


#ifndef BOOST_MATH_TEST_RATIONAL_HPP
#define BOOST_MATH_TEST_RATIONAL_HPP

#include <boost/test/tools/floating_point_comparison.hpp>
#include <boost/test/unit_test.hpp>
#include <boost/array.hpp>
#include <boost/math/tools/rational.hpp>
#include <boost/math/tools/precision.hpp>

template <class T, class U>
void do_test_spots1(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Everything past this point is generated by the program
   // ../tools/generate_rational_test.cpp
   //

   //
   // Polynomials of order 0
   //
   static const U n1c[1] = { 2 };
   static const boost::array<U, 1> n1a = {{ 2 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.125), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.25), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.75), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(1.0f - 1.0f/64.0f), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(6.5), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(10247.25), 1),
      static_cast<T>(0.2e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.125)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.25)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(0.75)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(6.5)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1c, static_cast<T>(10247.25)),
      static_cast<T>(0.2e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(0.125)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(0.25)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(0.75)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(6.5)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n1a, static_cast<T>(10247.25)),
      static_cast<T>(0.2e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.125), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.25), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.75), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(1.0f - 1.0f/64.0f), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(6.5f), 1),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(10247.25f), 1),
      static_cast<T>(0.2e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.125)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.25)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(0.75)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(6.5f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1c, static_cast<T>(10247.25f)),
      static_cast<T>(0.2e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(0.125)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(0.25)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(0.75)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(6.5f)),
      static_cast<T>(0.2e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n1a, static_cast<T>(10247.25f)),
      static_cast<T>(0.2e1L),
      tolerance);

   //
   // Rational functions of order 0
   //
   static const U d1c[1] = { 3 };
   static const boost::array<U, 1> d1a = {{ 3 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.125), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.25), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.75), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(1.0f - 1.0f/64.0f), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(6.5f), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(10247.25f), 1),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.125)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.25)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(0.75)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(6.5f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1c, d1c, static_cast<T>(10247.25f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(0.125)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(0.25)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(0.75)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(6.5f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n1a, d1a, static_cast<T>(10247.25f)),
      static_cast<T>(0.6666666666666666666666666666666666666667e0L),
      tolerance);

   //
   // Polynomials of order 1
   //
   static const U n2c[2] = { 3, 1 };
   static const boost::array<U, 2> n2a = {{ 3, 1 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.125), 2),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.25), 2),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.75), 2),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f), 2),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(6.5), 2),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(10247.25), 2),
      static_cast<T>(0.1025025e5L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.125)),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.25)),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(0.75)),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(6.5)),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2c, static_cast<T>(10247.25)),
      static_cast<T>(0.1025025e5L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(0.125)),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(0.25)),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(0.75)),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(6.5)),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n2a, static_cast<T>(10247.25)),
      static_cast<T>(0.1025025e5L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.125), 2),
      static_cast<T>(0.3015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.25), 2),
      static_cast<T>(0.30625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.75), 2),
      static_cast<T>(0.35625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f), 2),
      static_cast<T>(0.3968994140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(6.5f), 2),
      static_cast<T>(0.4525e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(10247.25f), 2),
      static_cast<T>(0.1050061355625e9L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.125)),
      static_cast<T>(0.3015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.25)),
      static_cast<T>(0.30625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(0.75)),
      static_cast<T>(0.35625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3968994140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(6.5f)),
      static_cast<T>(0.4525e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1050061355625e9L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(0.125)),
      static_cast<T>(0.3015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(0.25)),
      static_cast<T>(0.30625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(0.75)),
      static_cast<T>(0.35625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3968994140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(6.5f)),
      static_cast<T>(0.4525e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n2a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1050061355625e9L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.125), 2),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.25), 2),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.75), 2),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f), 2),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(6.5f), 2),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(10247.25f), 2),
      static_cast<T>(0.1025025e5L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.125)),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.25)),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(0.75)),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(6.5f)),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1025025e5L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(0.125)),
      static_cast<T>(0.3125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(0.25)),
      static_cast<T>(0.325e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(0.75)),
      static_cast<T>(0.375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(6.5f)),
      static_cast<T>(0.95e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n2a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1025025e5L),
      tolerance);

   //
   // Rational functions of order 1
   //
   static const U d2c[2] = { 5, 9 };
   static const boost::array<U, 2> d2a = {{ 5, 9 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.125), 2),
      static_cast<T>(0.5102040816326530612244897959183673469388e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.25), 2),
      static_cast<T>(0.4482758620689655172413793103448275862069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.75), 2),
      static_cast<T>(0.3191489361702127659574468085106382978723e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(1.0f - 1.0f/64.0f), 2),
      static_cast<T>(0.2874859075535512965050732807215332581736e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(6.5f), 2),
      static_cast<T>(0.1496062992125984251968503937007874015748e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(10247.25f), 2),
      static_cast<T>(0.1111376148281068304596377002122405609873e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.125)),
      static_cast<T>(0.5102040816326530612244897959183673469388e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.25)),
      static_cast<T>(0.4482758620689655172413793103448275862069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(0.75)),
      static_cast<T>(0.3191489361702127659574468085106382978723e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2874859075535512965050732807215332581736e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(6.5f)),
      static_cast<T>(0.1496062992125984251968503937007874015748e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2c, d2c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1111376148281068304596377002122405609873e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(0.125)),
      static_cast<T>(0.5102040816326530612244897959183673469388e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(0.25)),
      static_cast<T>(0.4482758620689655172413793103448275862069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(0.75)),
      static_cast<T>(0.3191489361702127659574468085106382978723e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.2874859075535512965050732807215332581736e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(6.5f)),
      static_cast<T>(0.1496062992125984251968503937007874015748e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n2a, d2a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1111376148281068304596377002122405609873e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots2(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 2
   //
   static const U n3c[3] = { 10, 6, 11 };
   static const boost::array<U, 3> n3a = {{ 10, 6, 11 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.125), 3),
      static_cast<T>(0.10921875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.25), 3),
      static_cast<T>(0.121875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.75), 3),
      static_cast<T>(0.206875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f), 3),
      static_cast<T>(0.26565185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(6.5), 3),
      static_cast<T>(0.51375e3L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(10247.25), 3),
      static_cast<T>(0.11551289516875e10L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.125)),
      static_cast<T>(0.10921875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.25)),
      static_cast<T>(0.121875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(0.75)),
      static_cast<T>(0.206875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26565185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(6.5)),
      static_cast<T>(0.51375e3L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3c, static_cast<T>(10247.25)),
      static_cast<T>(0.11551289516875e10L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(0.125)),
      static_cast<T>(0.10921875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(0.25)),
      static_cast<T>(0.121875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(0.75)),
      static_cast<T>(0.206875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26565185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(6.5)),
      static_cast<T>(0.51375e3L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n3a, static_cast<T>(10247.25)),
      static_cast<T>(0.11551289516875e10L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.125), 3),
      static_cast<T>(0.10096435546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.25), 3),
      static_cast<T>(0.1041796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.75), 3),
      static_cast<T>(0.1685546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f), 3),
      static_cast<T>(0.26142410933971405029296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(6.5f), 3),
      static_cast<T>(0.198991875e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(10247.25f), 3),
      static_cast<T>(0.12128916726310335635546875e18L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.125)),
      static_cast<T>(0.10096435546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.25)),
      static_cast<T>(0.1041796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(0.75)),
      static_cast<T>(0.1685546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26142410933971405029296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(6.5f)),
      static_cast<T>(0.198991875e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3c, static_cast<T>(10247.25f)),
      static_cast<T>(0.12128916726310335635546875e18L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(0.125)),
      static_cast<T>(0.10096435546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(0.25)),
      static_cast<T>(0.1041796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(0.75)),
      static_cast<T>(0.1685546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26142410933971405029296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(6.5f)),
      static_cast<T>(0.198991875e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n3a, static_cast<T>(10247.25f)),
      static_cast<T>(0.12128916726310335635546875e18L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.125), 3),
      static_cast<T>(0.10771484375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.25), 3),
      static_cast<T>(0.11671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.75), 3),
      static_cast<T>(0.19140625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f), 3),
      static_cast<T>(0.26398639678955078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(6.5f), 3),
      static_cast<T>(0.3069875e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(10247.25f), 3),
      static_cast<T>(0.11836265072405359375e14L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.125)),
      static_cast<T>(0.10771484375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.25)),
      static_cast<T>(0.11671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(0.75)),
      static_cast<T>(0.19140625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26398639678955078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(6.5f)),
      static_cast<T>(0.3069875e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3c, static_cast<T>(10247.25f)),
      static_cast<T>(0.11836265072405359375e14L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(0.125)),
      static_cast<T>(0.10771484375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(0.25)),
      static_cast<T>(0.11671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(0.75)),
      static_cast<T>(0.19140625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.26398639678955078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(6.5f)),
      static_cast<T>(0.3069875e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n3a, static_cast<T>(10247.25f)),
      static_cast<T>(0.11836265072405359375e14L),
      tolerance);

   //
   // Rational functions of order 2
   //
   static const U d3c[3] = { 3, 4, 10 };
   static const boost::array<U, 3> d3a = {{ 3, 4, 10 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.125), 3),
      static_cast<T>(0.2987179487179487179487179487179487179487e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.25), 3),
      static_cast<T>(0.2635135135135135135135135135135135135135e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.75), 3),
      static_cast<T>(0.1779569892473118279569892473118279569892e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(1.0f - 1.0f/64.0f), 3),
      static_cast<T>(0.1597671277126831703520981998649164537633e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(6.5f), 3),
      static_cast<T>(0.1137873754152823920265780730897009966777e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(10247.25f), 3),
      static_cast<T>(0.1100015619716026431429617996316152069115e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.125)),
      static_cast<T>(0.2987179487179487179487179487179487179487e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.25)),
      static_cast<T>(0.2635135135135135135135135135135135135135e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(0.75)),
      static_cast<T>(0.1779569892473118279569892473118279569892e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1597671277126831703520981998649164537633e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(6.5f)),
      static_cast<T>(0.1137873754152823920265780730897009966777e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3c, d3c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1100015619716026431429617996316152069115e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(0.125)),
      static_cast<T>(0.2987179487179487179487179487179487179487e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(0.25)),
      static_cast<T>(0.2635135135135135135135135135135135135135e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(0.75)),
      static_cast<T>(0.1779569892473118279569892473118279569892e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1597671277126831703520981998649164537633e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(6.5f)),
      static_cast<T>(0.1137873754152823920265780730897009966777e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n3a, d3a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1100015619716026431429617996316152069115e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots3(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 3
   //
   static const U n4c[4] = { 1, 4, 9, 11 };
   static const boost::array<U, 4> n4a = {{ 1, 4, 9, 11 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.125), 4),
      static_cast<T>(0.1662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.25), 4),
      static_cast<T>(0.2734375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.75), 4),
      static_cast<T>(0.13703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f), 4),
      static_cast<T>(0.24150836944580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(6.5), 4),
      static_cast<T>(0.3428125e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(10247.25), 4),
      static_cast<T>(0.11837210107094921875e14L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.125)),
      static_cast<T>(0.1662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.25)),
      static_cast<T>(0.2734375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(0.75)),
      static_cast<T>(0.13703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.24150836944580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(6.5)),
      static_cast<T>(0.3428125e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4c, static_cast<T>(10247.25)),
      static_cast<T>(0.11837210107094921875e14L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(0.125)),
      static_cast<T>(0.1662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(0.25)),
      static_cast<T>(0.2734375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(0.75)),
      static_cast<T>(0.13703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.24150836944580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(6.5)),
      static_cast<T>(0.3428125e4L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n4a, static_cast<T>(10247.25)),
      static_cast<T>(0.11837210107094921875e14L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.125), 4),
      static_cast<T>(0.1064739227294921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.25), 4),
      static_cast<T>(0.1287841796875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.75), 4),
      static_cast<T>(0.8055419921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f), 4),
      static_cast<T>(0.23334727106775972060859203338623046875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(6.5f), 4),
      static_cast<T>(0.845843359375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(10247.25f), 4),
      static_cast<T>(0.12736106409103529349764202508544921875e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.125)),
      static_cast<T>(0.1064739227294921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.25)),
      static_cast<T>(0.1287841796875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(0.75)),
      static_cast<T>(0.8055419921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.23334727106775972060859203338623046875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(6.5f)),
      static_cast<T>(0.845843359375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4c, static_cast<T>(10247.25f)),
      static_cast<T>(0.12736106409103529349764202508544921875e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(0.125)),
      static_cast<T>(0.1064739227294921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(0.25)),
      static_cast<T>(0.1287841796875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(0.75)),
      static_cast<T>(0.8055419921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.23334727106775972060859203338623046875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(6.5f)),
      static_cast<T>(0.845843359375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n4a, static_cast<T>(10247.25f)),
      static_cast<T>(0.12736106409103529349764202508544921875e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.125), 4),
      static_cast<T>(0.1517913818359375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.25), 4),
      static_cast<T>(0.21513671875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.75), 4),
      static_cast<T>(0.104072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f), 4),
      static_cast<T>(0.23689246584661304950714111328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(6.5f), 4),
      static_cast<T>(0.13013059375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(10247.25f), 4),
      static_cast<T>(0.12428804224649080826343115234375e22L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.125)),
      static_cast<T>(0.1517913818359375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.25)),
      static_cast<T>(0.21513671875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(0.75)),
      static_cast<T>(0.104072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.23689246584661304950714111328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(6.5f)),
      static_cast<T>(0.13013059375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4c, static_cast<T>(10247.25f)),
      static_cast<T>(0.12428804224649080826343115234375e22L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(0.125)),
      static_cast<T>(0.1517913818359375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(0.25)),
      static_cast<T>(0.21513671875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(0.75)),
      static_cast<T>(0.104072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.23689246584661304950714111328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(6.5f)),
      static_cast<T>(0.13013059375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n4a, static_cast<T>(10247.25f)),
      static_cast<T>(0.12428804224649080826343115234375e22L),
      tolerance);

   //
   // Rational functions of order 3
   //
   static const U d4c[4] = { 10, 2, 5, 4 };
   static const boost::array<U, 4> d4a = {{ 10, 2, 5, 4 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.125), 4),
      static_cast<T>(0.1608087679516250944822373393801965230537e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.25), 4),
      static_cast<T>(0.2514367816091954022988505747126436781609e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.75), 4),
      static_cast<T>(0.8564453125e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(1.0f - 1.0f/64.0f), 4),
      static_cast<T>(0.1170714951947222939292918160495461743806e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(6.5f), 4),
      static_cast<T>(0.2572219095854436315888201087975989495404e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(10247.25f), 4),
      static_cast<T>(0.2749884125808399380227005558292823797886e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.125)),
      static_cast<T>(0.1608087679516250944822373393801965230537e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.25)),
      static_cast<T>(0.2514367816091954022988505747126436781609e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(0.75)),
      static_cast<T>(0.8564453125e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1170714951947222939292918160495461743806e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(6.5f)),
      static_cast<T>(0.2572219095854436315888201087975989495404e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4c, d4c, static_cast<T>(10247.25f)),
      static_cast<T>(0.2749884125808399380227005558292823797886e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(0.125)),
      static_cast<T>(0.1608087679516250944822373393801965230537e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(0.25)),
      static_cast<T>(0.2514367816091954022988505747126436781609e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(0.75)),
      static_cast<T>(0.8564453125e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1170714951947222939292918160495461743806e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(6.5f)),
      static_cast<T>(0.2572219095854436315888201087975989495404e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n4a, d4a, static_cast<T>(10247.25f)),
      static_cast<T>(0.2749884125808399380227005558292823797886e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots4(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 4
   //
   static const U n5c[5] = { 10, 10, 4, 11, 9 };
   static const boost::array<U, 5> n5a = {{ 10, 10, 4, 11, 9 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.125), 5),
      static_cast<T>(0.11336181640625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.25), 5),
      static_cast<T>(0.1295703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.75), 5),
      static_cast<T>(0.2723828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f), 5),
      static_cast<T>(0.42662663042545318603515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(6.5), 5),
      static_cast<T>(0.193304375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(10247.25), 5),
      static_cast<T>(0.9924842756673782995703125e17L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.125)),
      static_cast<T>(0.11336181640625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.25)),
      static_cast<T>(0.1295703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(0.75)),
      static_cast<T>(0.2723828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.42662663042545318603515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(6.5)),
      static_cast<T>(0.193304375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5c, static_cast<T>(10247.25)),
      static_cast<T>(0.9924842756673782995703125e17L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(0.125)),
      static_cast<T>(0.11336181640625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(0.25)),
      static_cast<T>(0.1295703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(0.75)),
      static_cast<T>(0.2723828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.42662663042545318603515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(6.5)),
      static_cast<T>(0.193304375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n5a, static_cast<T>(10247.25)),
      static_cast<T>(0.9924842756673782995703125e17L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.125), 5),
      static_cast<T>(0.10157269060611724853515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.25), 5),
      static_cast<T>(0.106434478759765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.75), 5),
      static_cast<T>(0.197494049072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f), 5),
      static_cast<T>(0.4138858164296656028113829961512237787247e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(6.5f), 5),
      static_cast<T>(0.2951521370703125e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(10247.25f), 5),
      static_cast<T>(0.1094211231602999407223950000397888253311e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.125)),
      static_cast<T>(0.10157269060611724853515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.25)),
      static_cast<T>(0.106434478759765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(0.75)),
      static_cast<T>(0.197494049072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4138858164296656028113829961512237787247e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(6.5f)),
      static_cast<T>(0.2951521370703125e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1094211231602999407223950000397888253311e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(0.125)),
      static_cast<T>(0.10157269060611724853515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(0.25)),
      static_cast<T>(0.106434478759765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(0.75)),
      static_cast<T>(0.197494049072265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4138858164296656028113829961512237787247e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(6.5f)),
      static_cast<T>(0.2951521370703125e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n5a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1094211231602999407223950000397888253311e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.125), 5),
      static_cast<T>(0.11258152484893798828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.25), 5),
      static_cast<T>(0.1257379150390625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.75), 5),
      static_cast<T>(0.2299920654296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f), 5),
      static_cast<T>(0.4188681309761682314274366945028305053711e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(6.5f), 5),
      static_cast<T>(0.45408105703125e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(10247.25f), 5),
      static_cast<T>(0.106780963829612765105169679718983215332e30L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.125)),
      static_cast<T>(0.11258152484893798828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.25)),
      static_cast<T>(0.1257379150390625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(0.75)),
      static_cast<T>(0.2299920654296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4188681309761682314274366945028305053711e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(6.5f)),
      static_cast<T>(0.45408105703125e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5c, static_cast<T>(10247.25f)),
      static_cast<T>(0.106780963829612765105169679718983215332e30L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(0.125)),
      static_cast<T>(0.11258152484893798828125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(0.25)),
      static_cast<T>(0.1257379150390625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(0.75)),
      static_cast<T>(0.2299920654296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4188681309761682314274366945028305053711e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(6.5f)),
      static_cast<T>(0.45408105703125e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n5a, static_cast<T>(10247.25f)),
      static_cast<T>(0.106780963829612765105169679718983215332e30L),
      tolerance);

   //
   // Rational functions of order 4
   //
   static const U d5c[5] = { 6, 9, 6, 2, 5 };
   static const boost::array<U, 5> d5a = {{ 6, 9, 6, 2, 5 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.125), 5),
      static_cast<T>(0.1569265605461489066882963263374902835513e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.25), 5),
      static_cast<T>(0.1493471409275101305718144979738856371004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.75), 5),
      static_cast<T>(0.1468309117708991366603495472731101284481e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(1.0f - 1.0f/64.0f), 5),
      static_cast<T>(0.1564121691159921277310988862398683772017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(6.5f), 5),
      static_cast<T>(0.1973991741181125982091000185089449263153e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(10247.25f), 5),
      static_cast<T>(0.1800144410401676792233921448870747965702e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.125)),
      static_cast<T>(0.1569265605461489066882963263374902835513e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.25)),
      static_cast<T>(0.1493471409275101305718144979738856371004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(0.75)),
      static_cast<T>(0.1468309117708991366603495472731101284481e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1564121691159921277310988862398683772017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(6.5f)),
      static_cast<T>(0.1973991741181125982091000185089449263153e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5c, d5c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1800144410401676792233921448870747965702e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(0.125)),
      static_cast<T>(0.1569265605461489066882963263374902835513e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(0.25)),
      static_cast<T>(0.1493471409275101305718144979738856371004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(0.75)),
      static_cast<T>(0.1468309117708991366603495472731101284481e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1564121691159921277310988862398683772017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(6.5f)),
      static_cast<T>(0.1973991741181125982091000185089449263153e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n5a, d5a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1800144410401676792233921448870747965702e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots5(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 5
   //
   static const U n6c[6] = { 6, 8, 12, 5, 7, 5 };
   static const boost::array<U, 6> n6a = {{ 6, 8, 12, 5, 7, 5 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.125), 6),
      static_cast<T>(0.7199127197265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.25), 6),
      static_cast<T>(0.88603515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.75), 6),
      static_cast<T>(0.242607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f), 6),
      static_cast<T>(0.41466238017193973064422607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(6.5), 6),
      static_cast<T>(0.7244809375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(10247.25), 6),
      static_cast<T>(0.5650228315695522094919501953125e21L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.125)),
      static_cast<T>(0.7199127197265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.25)),
      static_cast<T>(0.88603515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(0.75)),
      static_cast<T>(0.242607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.41466238017193973064422607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(6.5)),
      static_cast<T>(0.7244809375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6c, static_cast<T>(10247.25)),
      static_cast<T>(0.5650228315695522094919501953125e21L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(0.125)),
      static_cast<T>(0.7199127197265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(0.25)),
      static_cast<T>(0.88603515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(0.75)),
      static_cast<T>(0.242607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.41466238017193973064422607421875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(6.5)),
      static_cast<T>(0.7244809375e5L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n6a, static_cast<T>(10247.25)),
      static_cast<T>(0.5650228315695522094919501953125e21L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.125), 6),
      static_cast<T>(0.6127949182875454425811767578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.25), 6),
      static_cast<T>(0.654820728302001953125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.75), 6),
      static_cast<T>(0.1616912555694580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f), 6),
      static_cast<T>(0.4001137167344577683526091194110563264985e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(6.5f), 6),
      static_cast<T>(0.6958411633369140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.125)),
      static_cast<T>(0.6127949182875454425811767578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.25)),
      static_cast<T>(0.654820728302001953125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(0.75)),
      static_cast<T>(0.1616912555694580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4001137167344577683526091194110563264985e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6c, static_cast<T>(6.5f)),
      static_cast<T>(0.6958411633369140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6a, static_cast<T>(0.125)),
      static_cast<T>(0.6127949182875454425811767578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6a, static_cast<T>(0.25)),
      static_cast<T>(0.654820728302001953125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6a, static_cast<T>(0.75)),
      static_cast<T>(0.1616912555694580078125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4001137167344577683526091194110563264985e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n6a, static_cast<T>(6.5f)),
      static_cast<T>(0.6958411633369140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.125), 6),
      static_cast<T>(0.7023593463003635406494140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.25), 6),
      static_cast<T>(0.8192829132080078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.75), 6),
      static_cast<T>(0.19558834075927734375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f), 6),
      static_cast<T>(0.4055123471588142408661425974969461094588e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(6.5f), 6),
      static_cast<T>(0.107052491744140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(10247.25f), 6),
      static_cast<T>(0.6229253367792843599034768117351560896265e37L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.125)),
      static_cast<T>(0.7023593463003635406494140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.25)),
      static_cast<T>(0.8192829132080078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(0.75)),
      static_cast<T>(0.19558834075927734375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4055123471588142408661425974969461094588e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(6.5f)),
      static_cast<T>(0.107052491744140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6c, static_cast<T>(10247.25f)),
      static_cast<T>(0.6229253367792843599034768117351560896265e37L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(0.125)),
      static_cast<T>(0.7023593463003635406494140625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(0.25)),
      static_cast<T>(0.8192829132080078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(0.75)),
      static_cast<T>(0.19558834075927734375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4055123471588142408661425974969461094588e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(6.5f)),
      static_cast<T>(0.107052491744140625e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n6a, static_cast<T>(10247.25f)),
      static_cast<T>(0.6229253367792843599034768117351560896265e37L),
      tolerance);

   //
   // Rational functions of order 5
   //
   static const U d6c[6] = { 5, 11, 7, 12, 10, 5 };
   static const boost::array<U, 6> d6a = {{ 5, 11, 7, 12, 10, 5 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.125), 6),
      static_cast<T>(0.1105787665293227020667219792530925829572e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.25), 6),
      static_cast<T>(0.1052430112515949425820670455863588910799e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.75), 6),
      static_cast<T>(0.9120378868534087154447666948125848966555e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(1.0f - 1.0f/64.0f), 6),
      static_cast<T>(0.8626539746676637108178973543021882567334e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(6.5f), 6),
      static_cast<T>(0.9109197333022141385660730837432462113756e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(10247.25f), 6),
      static_cast<T>(0.9999414458034701919327379302959430043671e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.125)),
      static_cast<T>(0.1105787665293227020667219792530925829572e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.25)),
      static_cast<T>(0.1052430112515949425820670455863588910799e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(0.75)),
      static_cast<T>(0.9120378868534087154447666948125848966555e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8626539746676637108178973543021882567334e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(6.5f)),
      static_cast<T>(0.9109197333022141385660730837432462113756e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6c, d6c, static_cast<T>(10247.25f)),
      static_cast<T>(0.9999414458034701919327379302959430043671e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(0.125)),
      static_cast<T>(0.1105787665293227020667219792530925829572e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(0.25)),
      static_cast<T>(0.1052430112515949425820670455863588910799e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(0.75)),
      static_cast<T>(0.9120378868534087154447666948125848966555e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8626539746676637108178973543021882567334e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(6.5f)),
      static_cast<T>(0.9109197333022141385660730837432462113756e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n6a, d6a, static_cast<T>(10247.25f)),
      static_cast<T>(0.9999414458034701919327379302959430043671e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots6(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 6
   //
   static const U n7c[7] = { 3, 4, 11, 5, 10, 7, 9 };
   static const boost::array<U, 7> n7a = {{ 3, 4, 11, 5, 10, 7, 9 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.125), 7),
      static_cast<T>(0.3684329986572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.25), 7),
      static_cast<T>(0.4813720703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.75), 7),
      static_cast<T>(0.20723876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f), 7),
      static_cast<T>(0.46413680258337990380823612213134765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(6.5), 7),
      static_cast<T>(0.779707859375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(10247.25), 7),
      static_cast<T>(0.10421241651331160693970241510986328125e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.125)),
      static_cast<T>(0.3684329986572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.25)),
      static_cast<T>(0.4813720703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(0.75)),
      static_cast<T>(0.20723876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.46413680258337990380823612213134765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(6.5)),
      static_cast<T>(0.779707859375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7c, static_cast<T>(10247.25)),
      static_cast<T>(0.10421241651331160693970241510986328125e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(0.125)),
      static_cast<T>(0.3684329986572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(0.25)),
      static_cast<T>(0.4813720703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(0.75)),
      static_cast<T>(0.20723876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.46413680258337990380823612213134765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(6.5)),
      static_cast<T>(0.779707859375e6L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n7a, static_cast<T>(10247.25)),
      static_cast<T>(0.10421241651331160693970241510986328125e26L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.125), 7),
      static_cast<T>(0.3065205223058001138269901275634765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.25), 7),
      static_cast<T>(0.3294349253177642822265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.75), 7),
      static_cast<T>(0.11300772249698638916015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f), 7),
      static_cast<T>(0.4400013192129567626077980251194602528964e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(6.5f), 7),
      static_cast<T>(0.52166734985505126953125e11L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.125)),
      static_cast<T>(0.3065205223058001138269901275634765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.25)),
      static_cast<T>(0.3294349253177642822265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(0.75)),
      static_cast<T>(0.11300772249698638916015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4400013192129567626077980251194602528964e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7c, static_cast<T>(6.5f)),
      static_cast<T>(0.52166734985505126953125e11L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7a, static_cast<T>(0.125)),
      static_cast<T>(0.3065205223058001138269901275634765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7a, static_cast<T>(0.25)),
      static_cast<T>(0.3294349253177642822265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7a, static_cast<T>(0.75)),
      static_cast<T>(0.11300772249698638916015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4400013192129567626077980251194602528964e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n7a, static_cast<T>(6.5f)),
      static_cast<T>(0.52166734985505126953125e11L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.125), 7),
      static_cast<T>(0.3521641784464009106159210205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.25), 7),
      static_cast<T>(0.41773970127105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.75), 7),
      static_cast<T>(0.140676963329315185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f), 7),
      static_cast<T>(0.4465092766607814731253821207562770823074e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(6.5f), 7),
      static_cast<T>(0.802565153877001953125e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.125)),
      static_cast<T>(0.3521641784464009106159210205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.25)),
      static_cast<T>(0.41773970127105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(0.75)),
      static_cast<T>(0.140676963329315185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4465092766607814731253821207562770823074e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7c, static_cast<T>(6.5f)),
      static_cast<T>(0.802565153877001953125e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7a, static_cast<T>(0.125)),
      static_cast<T>(0.3521641784464009106159210205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7a, static_cast<T>(0.25)),
      static_cast<T>(0.41773970127105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7a, static_cast<T>(0.75)),
      static_cast<T>(0.140676963329315185546875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4465092766607814731253821207562770823074e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n7a, static_cast<T>(6.5f)),
      static_cast<T>(0.802565153877001953125e10L),
      tolerance);
   //
   // Rational functions of order 6
   //
   static const U d7c[7] = { 2, 8, 10, 8, 1, 11, 1 };
   static const boost::array<U, 7> d7a = {{ 2, 8, 10, 8, 1, 11, 1 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.125), 7),
      static_cast<T>(0.1161348466465698540596242849979738853664e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.25), 7),
      static_cast<T>(0.1010247476558897371522262642824204539632e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.75), 7),
      static_cast<T>(0.103079575951134804308491906398377636644e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(1.0f - 1.0f/64.0f), 7),
      static_cast<T>(0.1183671559390403425417413542214181025989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(6.5f), 7),
      static_cast<T>(0.3757457624476840396478985178609736319955e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(10247.25f), 7),
      static_cast<T>(0.8991031618241406513349732955332873069556e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.125)),
      static_cast<T>(0.1161348466465698540596242849979738853664e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.25)),
      static_cast<T>(0.1010247476558897371522262642824204539632e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(0.75)),
      static_cast<T>(0.103079575951134804308491906398377636644e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1183671559390403425417413542214181025989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(6.5f)),
      static_cast<T>(0.3757457624476840396478985178609736319955e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7c, d7c, static_cast<T>(10247.25f)),
      static_cast<T>(0.8991031618241406513349732955332873069556e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(0.125)),
      static_cast<T>(0.1161348466465698540596242849979738853664e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(0.25)),
      static_cast<T>(0.1010247476558897371522262642824204539632e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(0.75)),
      static_cast<T>(0.103079575951134804308491906398377636644e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1183671559390403425417413542214181025989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(6.5f)),
      static_cast<T>(0.3757457624476840396478985178609736319955e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n7a, d7a, static_cast<T>(10247.25f)),
      static_cast<T>(0.8991031618241406513349732955332873069556e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots7(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 7
   //
   static const U n8c[8] = { 9, 5, 6, 1, 12, 2, 11, 1 };
   static const boost::array<U, 8> n8a = {{ 9, 5, 6, 1, 12, 2, 11, 1 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.125), 8),
      static_cast<T>(0.9723736286163330078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.25), 8),
      static_cast<T>(0.1069219970703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.75), 8),
      static_cast<T>(0.2290960693359375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f), 8),
      static_cast<T>(0.4470947054706607559637632220983505249023e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(6.5), 8),
      static_cast<T>(0.13650267734375e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(10247.25), 8),
      static_cast<T>(0.1187728773094625678513681460864459228516e29L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.125)),
      static_cast<T>(0.9723736286163330078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.25)),
      static_cast<T>(0.1069219970703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(0.75)),
      static_cast<T>(0.2290960693359375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4470947054706607559637632220983505249023e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(6.5)),
      static_cast<T>(0.13650267734375e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8c, static_cast<T>(10247.25)),
      static_cast<T>(0.1187728773094625678513681460864459228516e29L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(0.125)),
      static_cast<T>(0.9723736286163330078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(0.25)),
      static_cast<T>(0.1069219970703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(0.75)),
      static_cast<T>(0.2290960693359375e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4470947054706607559637632220983505249023e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(6.5)),
      static_cast<T>(0.13650267734375e7L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n8a, static_cast<T>(10247.25)),
      static_cast<T>(0.1187728773094625678513681460864459228516e29L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.125), 8),
      static_cast<T>(0.9079594375725946520105935633182525634766e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.25), 8),
      static_cast<T>(0.93363673128187656402587890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.75), 8),
      static_cast<T>(0.155691558457911014556884765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f), 8),
      static_cast<T>(0.4258457138999176910226338119632657870077e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(6.5f), 8),
      static_cast<T>(0.30319406120433428955078125e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.125)),
      static_cast<T>(0.9079594375725946520105935633182525634766e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.25)),
      static_cast<T>(0.93363673128187656402587890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(0.75)),
      static_cast<T>(0.155691558457911014556884765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4258457138999176910226338119632657870077e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8c, static_cast<T>(6.5f)),
      static_cast<T>(0.30319406120433428955078125e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8a, static_cast<T>(0.125)),
      static_cast<T>(0.9079594375725946520105935633182525634766e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8a, static_cast<T>(0.25)),
      static_cast<T>(0.93363673128187656402587890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8a, static_cast<T>(0.75)),
      static_cast<T>(0.155691558457911014556884765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4258457138999176910226338119632657870077e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n8a, static_cast<T>(6.5f)),
      static_cast<T>(0.30319406120433428955078125e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.125), 8),
      static_cast<T>(0.9636755005807572160847485065460205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.25), 8),
      static_cast<T>(0.1034546925127506256103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.75), 8),
      static_cast<T>(0.1775887446105480194091796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f), 8),
      static_cast<T>(0.4311765982475354321499772058039525455316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(6.5f), 8),
      static_cast<T>(0.466452401928975830078125e11L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.125)),
      static_cast<T>(0.9636755005807572160847485065460205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.25)),
      static_cast<T>(0.1034546925127506256103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(0.75)),
      static_cast<T>(0.1775887446105480194091796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4311765982475354321499772058039525455316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8c, static_cast<T>(6.5f)),
      static_cast<T>(0.466452401928975830078125e11L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8a, static_cast<T>(0.125)),
      static_cast<T>(0.9636755005807572160847485065460205078125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8a, static_cast<T>(0.25)),
      static_cast<T>(0.1034546925127506256103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8a, static_cast<T>(0.75)),
      static_cast<T>(0.1775887446105480194091796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4311765982475354321499772058039525455316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n8a, static_cast<T>(6.5f)),
      static_cast<T>(0.466452401928975830078125e11L),
      tolerance);
   //
   // Rational functions of order 7
   //
   static const U d8c[8] = { 7, 10, 10, 11, 2, 4, 1, 7 };
   static const boost::array<U, 8> d8a = {{ 7, 10, 10, 11, 2, 4, 1, 7 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.125), 8),
      static_cast<T>(0.1153693678861771369296601206130394355814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.25), 8),
      static_cast<T>(0.103714470093009762768860970830101771981e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.75), 8),
      static_cast<T>(0.834289461108456229648481346061946410909e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(1.0f - 1.0f/64.0f), 8),
      static_cast<T>(0.8981362736388035283180082845549732926578e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(6.5f), 8),
      static_cast<T>(0.383383275200627223098571833372266642165e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(10247.25f), 8),
      static_cast<T>(0.1430085023641929377426860779365964126311e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.125)),
      static_cast<T>(0.1153693678861771369296601206130394355814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.25)),
      static_cast<T>(0.103714470093009762768860970830101771981e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(0.75)),
      static_cast<T>(0.834289461108456229648481346061946410909e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8981362736388035283180082845549732926578e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(6.5f)),
      static_cast<T>(0.383383275200627223098571833372266642165e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8c, d8c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1430085023641929377426860779365964126311e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(0.125)),
      static_cast<T>(0.1153693678861771369296601206130394355814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(0.25)),
      static_cast<T>(0.103714470093009762768860970830101771981e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(0.75)),
      static_cast<T>(0.834289461108456229648481346061946410909e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8981362736388035283180082845549732926578e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(6.5f)),
      static_cast<T>(0.383383275200627223098571833372266642165e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n8a, d8a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1430085023641929377426860779365964126311e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots8(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 8
   //
   static const U n9c[9] = { 3, 9, 3, 9, 5, 6, 10, 7, 10 };
   static const boost::array<U, 9> n9a = {{ 3, 9, 3, 9, 5, 6, 10, 7, 10 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.125), 9),
      static_cast<T>(0.419089901447296142578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.25), 9),
      static_cast<T>(0.5606536865234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.75), 9),
      static_cast<T>(0.21955535888671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f), 9),
      static_cast<T>(0.577754343750660055434309470001608133316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(6.5), 9),
      static_cast<T>(0.3613143234375e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(10247.25), 9),
      static_cast<T>(0.1215873306624182859977656082228297326996e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.125)),
      static_cast<T>(0.419089901447296142578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.25)),
      static_cast<T>(0.5606536865234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(0.75)),
      static_cast<T>(0.21955535888671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.577754343750660055434309470001608133316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(6.5)),
      static_cast<T>(0.3613143234375e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9c, static_cast<T>(10247.25)),
      static_cast<T>(0.1215873306624182859977656082228297326996e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(0.125)),
      static_cast<T>(0.419089901447296142578125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(0.25)),
      static_cast<T>(0.5606536865234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(0.75)),
      static_cast<T>(0.21955535888671875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.577754343750660055434309470001608133316e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(6.5)),
      static_cast<T>(0.3613143234375e8L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n9a, static_cast<T>(10247.25)),
      static_cast<T>(0.1215873306624182859977656082228297326996e34L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.125), 9),
      static_cast<T>(0.3141392057908696244794555241242051124573e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.25), 9),
      static_cast<T>(0.35764986560679972171783447265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.75), 9),
      static_cast<T>(0.119936861679889261722564697265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f), 9),
      static_cast<T>(0.5392583642412261279423815221065846904997e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(6.5f), 9),
      static_cast<T>(0.103274449934495763275146484375e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.125)),
      static_cast<T>(0.3141392057908696244794555241242051124573e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.25)),
      static_cast<T>(0.35764986560679972171783447265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(0.75)),
      static_cast<T>(0.119936861679889261722564697265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5392583642412261279423815221065846904997e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9c, static_cast<T>(6.5f)),
      static_cast<T>(0.103274449934495763275146484375e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9a, static_cast<T>(0.125)),
      static_cast<T>(0.3141392057908696244794555241242051124573e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9a, static_cast<T>(0.25)),
      static_cast<T>(0.35764986560679972171783447265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9a, static_cast<T>(0.75)),
      static_cast<T>(0.119936861679889261722564697265625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5392583642412261279423815221065846904997e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n9a, static_cast<T>(6.5f)),
      static_cast<T>(0.103274449934495763275146484375e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.125), 9),
      static_cast<T>(0.4131136463269569958356441929936408996582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.25), 9),
      static_cast<T>(0.530599462427198886871337890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.75), 9),
      static_cast<T>(0.1499158155731856822967529296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f), 9),
      static_cast<T>(0.5473418303402932093382923399178003205076e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(6.5f), 9),
      static_cast<T>(0.1588837691300188665771484375e14L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.125)),
      static_cast<T>(0.4131136463269569958356441929936408996582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.25)),
      static_cast<T>(0.530599462427198886871337890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(0.75)),
      static_cast<T>(0.1499158155731856822967529296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5473418303402932093382923399178003205076e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9c, static_cast<T>(6.5f)),
      static_cast<T>(0.1588837691300188665771484375e14L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9a, static_cast<T>(0.125)),
      static_cast<T>(0.4131136463269569958356441929936408996582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9a, static_cast<T>(0.25)),
      static_cast<T>(0.530599462427198886871337890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9a, static_cast<T>(0.75)),
      static_cast<T>(0.1499158155731856822967529296875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5473418303402932093382923399178003205076e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n9a, static_cast<T>(6.5f)),
      static_cast<T>(0.1588837691300188665771484375e14L),
      tolerance);
   //
   // Rational functions of order 8
   //
   static const U d9c[9] = { 12, 3, 10, 4, 6, 6, 6, 10, 7 };
   static const boost::array<U, 9> d9a = {{ 12, 3, 10, 4, 6, 6, 6, 10, 7 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.125), 9),
      static_cast<T>(0.3341827920887278954826517708316980450243e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.25), 9),
      static_cast<T>(0.4162555242250224594229316089330159747956e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.75), 9),
      static_cast<T>(0.7844550246723573100342976024389406178802e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(1.0f - 1.0f/64.0f), 9),
      static_cast<T>(0.959335028097323235424759017468360386113e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(6.5f), 9),
      static_cast<T>(0.1302420407483849169746727326286868117535e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(10247.25f), 9),
      static_cast<T>(0.1428469874366314841691622991213446692856e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.125)),
      static_cast<T>(0.3341827920887278954826517708316980450243e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.25)),
      static_cast<T>(0.4162555242250224594229316089330159747956e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(0.75)),
      static_cast<T>(0.7844550246723573100342976024389406178802e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.959335028097323235424759017468360386113e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(6.5f)),
      static_cast<T>(0.1302420407483849169746727326286868117535e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9c, d9c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1428469874366314841691622991213446692856e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(0.125)),
      static_cast<T>(0.3341827920887278954826517708316980450243e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(0.25)),
      static_cast<T>(0.4162555242250224594229316089330159747956e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(0.75)),
      static_cast<T>(0.7844550246723573100342976024389406178802e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.959335028097323235424759017468360386113e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(6.5f)),
      static_cast<T>(0.1302420407483849169746727326286868117535e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n9a, d9a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1428469874366314841691622991213446692856e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots9(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 9
   //
   static const U n10c[10] = { 3, 4, 2, 6, 8, 1, 2, 4, 8, 8 };
   static const boost::array<U, 10> n10a = {{ 3, 4, 2, 6, 8, 1, 2, 4, 8, 8 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.125), 10),
      static_cast<T>(0.3544962465763092041015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.25), 10),
      static_cast<T>(0.4251861572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.75), 10),
      static_cast<T>(0.14716278076171875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f), 10),
      static_cast<T>(0.4243246072286939307716124858416151255369e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(6.5), 10),
      static_cast<T>(0.193326261328125e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(10247.25), 10),
      static_cast<T>(0.9967777935240642903307419028007759631098e37L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.125)),
      static_cast<T>(0.3544962465763092041015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.25)),
      static_cast<T>(0.4251861572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(0.75)),
      static_cast<T>(0.14716278076171875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4243246072286939307716124858416151255369e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(6.5)),
      static_cast<T>(0.193326261328125e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10c, static_cast<T>(10247.25)),
      static_cast<T>(0.9967777935240642903307419028007759631098e37L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(0.125)),
      static_cast<T>(0.3544962465763092041015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(0.25)),
      static_cast<T>(0.4251861572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(0.75)),
      static_cast<T>(0.14716278076171875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.4243246072286939307716124858416151255369e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(6.5)),
      static_cast<T>(0.193326261328125e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n10a, static_cast<T>(10247.25)),
      static_cast<T>(0.9967777935240642903307419028007759631098e37L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.125), 10),
      static_cast<T>(0.3063011647232116718697625401546247303486e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.25), 10),
      static_cast<T>(0.3259400503826327621936798095703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.75), 10),
      static_cast<T>(0.8067807371844537556171417236328125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f), 10),
      static_cast<T>(0.3922779295817342542834377568121069117613e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(6.5f), 10),
      static_cast<T>(0.3514067090785774022613525390625e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.125)),
      static_cast<T>(0.3063011647232116718697625401546247303486e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.25)),
      static_cast<T>(0.3259400503826327621936798095703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(0.75)),
      static_cast<T>(0.8067807371844537556171417236328125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3922779295817342542834377568121069117613e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10c, static_cast<T>(6.5f)),
      static_cast<T>(0.3514067090785774022613525390625e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10a, static_cast<T>(0.125)),
      static_cast<T>(0.3063011647232116718697625401546247303486e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10a, static_cast<T>(0.25)),
      static_cast<T>(0.3259400503826327621936798095703125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10a, static_cast<T>(0.75)),
      static_cast<T>(0.8067807371844537556171417236328125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3922779295817342542834377568121069117613e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n10a, static_cast<T>(6.5f)),
      static_cast<T>(0.3514067090785774022613525390625e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.125), 10),
      static_cast<T>(0.3504093177856933749581003212369978427887e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.25), 10),
      static_cast<T>(0.40376020153053104877471923828125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.75), 10),
      static_cast<T>(0.97570764957927167415618896484375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f), 10),
      static_cast<T>(0.3980283729084284487958732767615054341702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(6.5f), 10),
      static_cast<T>(0.54062570627473700347900390625e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.125)),
      static_cast<T>(0.3504093177856933749581003212369978427887e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.25)),
      static_cast<T>(0.40376020153053104877471923828125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(0.75)),
      static_cast<T>(0.97570764957927167415618896484375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3980283729084284487958732767615054341702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10c, static_cast<T>(6.5f)),
      static_cast<T>(0.54062570627473700347900390625e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10a, static_cast<T>(0.125)),
      static_cast<T>(0.3504093177856933749581003212369978427887e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10a, static_cast<T>(0.25)),
      static_cast<T>(0.40376020153053104877471923828125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10a, static_cast<T>(0.75)),
      static_cast<T>(0.97570764957927167415618896484375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.3980283729084284487958732767615054341702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n10a, static_cast<T>(6.5f)),
      static_cast<T>(0.54062570627473700347900390625e15L),
      tolerance);
   //
   // Rational functions of order 9
   //
   static const U d10c[10] = { 3, 11, 1, 12, 8, 8, 7, 10, 8, 8 };
   static const boost::array<U, 10> d10a = {{ 3, 11, 1, 12, 8, 8, 7, 10, 8, 8 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.125), 10),
      static_cast<T>(0.8027011456035955638016996485812217742707e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.25), 10),
      static_cast<T>(0.7037718026559713894599659542655668311705e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.75), 10),
      static_cast<T>(0.5819711007563332500606442409694389231031e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(1.0f - 1.0f/64.0f), 10),
      static_cast<T>(0.6021345078884753739086911192013079405483e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(6.5f), 10),
      static_cast<T>(0.9827105949744728065574239430024037810213e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(10247.25f), 10),
      static_cast<T>(0.9999999928576761766405011572543053028339e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.125)),
      static_cast<T>(0.8027011456035955638016996485812217742707e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.25)),
      static_cast<T>(0.7037718026559713894599659542655668311705e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(0.75)),
      static_cast<T>(0.5819711007563332500606442409694389231031e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6021345078884753739086911192013079405483e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(6.5f)),
      static_cast<T>(0.9827105949744728065574239430024037810213e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10c, d10c, static_cast<T>(10247.25f)),
      static_cast<T>(0.9999999928576761766405011572543053028339e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(0.125)),
      static_cast<T>(0.8027011456035955638016996485812217742707e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(0.25)),
      static_cast<T>(0.7037718026559713894599659542655668311705e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(0.75)),
      static_cast<T>(0.5819711007563332500606442409694389231031e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6021345078884753739086911192013079405483e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(6.5f)),
      static_cast<T>(0.9827105949744728065574239430024037810213e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n10a, d10a, static_cast<T>(10247.25f)),
      static_cast<T>(0.9999999928576761766405011572543053028339e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots10(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 10
   //
   static const U n11c[11] = { 2, 2, 8, 11, 3, 4, 10, 11, 5, 1, 6 };
   static const boost::array<U, 11> n11a = {{ 2, 2, 8, 11, 3, 4, 10, 11, 5, 1, 6 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.125), 11),
      static_cast<T>(0.239738257043063640594482421875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.25), 11),
      static_cast<T>(0.31906986236572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.75), 11),
      static_cast<T>(0.187007007598876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f), 11),
      static_cast<T>(0.5807897685780276847943015550157497273176e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(6.5), 11),
      static_cast<T>(0.85061053443359375e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.125)),
      static_cast<T>(0.239738257043063640594482421875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.25)),
      static_cast<T>(0.31906986236572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(0.75)),
      static_cast<T>(0.187007007598876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5807897685780276847943015550157497273176e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11c, static_cast<T>(6.5)),
      static_cast<T>(0.85061053443359375e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11a, static_cast<T>(0.125)),
      static_cast<T>(0.239738257043063640594482421875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11a, static_cast<T>(0.25)),
      static_cast<T>(0.31906986236572265625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11a, static_cast<T>(0.75)),
      static_cast<T>(0.187007007598876953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5807897685780276847943015550157497273176e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n11a, static_cast<T>(6.5)),
      static_cast<T>(0.85061053443359375e9L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.125), 11),
      static_cast<T>(0.2033245269357184586631048794913567689946e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.25), 11),
      static_cast<T>(0.2158985776148256263695657253265380859375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.75), 11),
      static_cast<T>(0.8727145384755203849636018276214599609375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f), 11),
      static_cast<T>(0.5363972553738812062759598952966094072427e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(6.5f), 11),
      static_cast<T>(0.1092297265410211371166019439697265625e18L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.125)),
      static_cast<T>(0.2033245269357184586631048794913567689946e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.25)),
      static_cast<T>(0.2158985776148256263695657253265380859375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(0.75)),
      static_cast<T>(0.8727145384755203849636018276214599609375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5363972553738812062759598952966094072427e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11c, static_cast<T>(6.5f)),
      static_cast<T>(0.1092297265410211371166019439697265625e18L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11a, static_cast<T>(0.125)),
      static_cast<T>(0.2033245269357184586631048794913567689946e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11a, static_cast<T>(0.25)),
      static_cast<T>(0.2158985776148256263695657253265380859375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11a, static_cast<T>(0.75)),
      static_cast<T>(0.8727145384755203849636018276214599609375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.5363972553738812062759598952966094072427e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n11a, static_cast<T>(6.5f)),
      static_cast<T>(0.1092297265410211371166019439697265625e18L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.125), 11),
      static_cast<T>(0.2265962154857476693048390359308541519567e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.25), 11),
      static_cast<T>(0.26359431045930250547826290130615234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.75), 11),
      static_cast<T>(0.109695271796736051328480243682861328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f), 11),
      static_cast<T>(0.544594037205212653994625925380682572437e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(6.5f), 11),
      static_cast<T>(0.16804573314003253556400299072265625e17L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.125)),
      static_cast<T>(0.2265962154857476693048390359308541519567e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.25)),
      static_cast<T>(0.26359431045930250547826290130615234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(0.75)),
      static_cast<T>(0.109695271796736051328480243682861328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.544594037205212653994625925380682572437e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11c, static_cast<T>(6.5f)),
      static_cast<T>(0.16804573314003253556400299072265625e17L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11a, static_cast<T>(0.125)),
      static_cast<T>(0.2265962154857476693048390359308541519567e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11a, static_cast<T>(0.25)),
      static_cast<T>(0.26359431045930250547826290130615234375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11a, static_cast<T>(0.75)),
      static_cast<T>(0.109695271796736051328480243682861328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.544594037205212653994625925380682572437e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n11a, static_cast<T>(6.5f)),
      static_cast<T>(0.16804573314003253556400299072265625e17L),
      tolerance);
   //
   // Rational functions of order 10
   //
   static const U d11c[11] = { 4, 1, 3, 9, 11, 8, 11, 2, 6, 6, 4 };
   static const boost::array<U, 11> d11a = {{ 4, 1, 3, 9, 11, 8, 11, 2, 6, 6, 4 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.125), 11),
      static_cast<T>(0.5718365676248588095654568811483084403598e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.25), 11),
      static_cast<T>(0.6888631839546304707516922567568791812269e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.75), 11),
      static_cast<T>(0.9783539912974912482969079012097816310129e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(1.0f - 1.0f/64.0f), 11),
      static_cast<T>(0.9694017102874332007392886881642471036972e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(6.5f), 11),
      static_cast<T>(0.1243900864392932237542421996384079347041e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(10247.25f), 11),
      static_cast<T>(0.1499804844733304585200728061706913399511e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.125)),
      static_cast<T>(0.5718365676248588095654568811483084403598e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.25)),
      static_cast<T>(0.6888631839546304707516922567568791812269e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(0.75)),
      static_cast<T>(0.9783539912974912482969079012097816310129e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9694017102874332007392886881642471036972e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(6.5f)),
      static_cast<T>(0.1243900864392932237542421996384079347041e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11c, d11c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1499804844733304585200728061706913399511e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(0.125)),
      static_cast<T>(0.5718365676248588095654568811483084403598e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(0.25)),
      static_cast<T>(0.6888631839546304707516922567568791812269e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(0.75)),
      static_cast<T>(0.9783539912974912482969079012097816310129e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9694017102874332007392886881642471036972e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(6.5f)),
      static_cast<T>(0.1243900864392932237542421996384079347041e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n11a, d11a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1499804844733304585200728061706913399511e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots11(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 11
   //
   static const U n12c[12] = { 10, 12, 4, 1, 12, 7, 11, 5, 12, 5, 10, 6 };
   static const boost::array<U, 12> n12a = {{ 10, 12, 4, 1, 12, 7, 11, 5, 12, 5, 10, 6 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.125), 12),
      static_cast<T>(0.1156764154392294585704803466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.25), 12),
      static_cast<T>(0.13322539806365966796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.75), 12),
      static_cast<T>(0.32148390293121337890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f), 12),
      static_cast<T>(0.8737331822870016474402916385744166660743e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(6.5), 12),
      static_cast<T>(0.67419250750654296875e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.125)),
      static_cast<T>(0.1156764154392294585704803466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.25)),
      static_cast<T>(0.13322539806365966796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(0.75)),
      static_cast<T>(0.32148390293121337890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8737331822870016474402916385744166660743e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12c, static_cast<T>(6.5)),
      static_cast<T>(0.67419250750654296875e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12a, static_cast<T>(0.125)),
      static_cast<T>(0.1156764154392294585704803466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12a, static_cast<T>(0.25)),
      static_cast<T>(0.13322539806365966796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12a, static_cast<T>(0.75)),
      static_cast<T>(0.32148390293121337890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8737331822870016474402916385744166660743e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n12a, static_cast<T>(6.5)),
      static_cast<T>(0.67419250750654296875e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.125), 12),
      static_cast<T>(0.101884810991335118067599354446661763518e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.25), 12),
      static_cast<T>(0.1076605959896767217287560924887657165527e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.75), 12),
      static_cast<T>(0.2041755737073560794669901952147483825684e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f), 12),
      static_cast<T>(0.8060387357327405373376954971672611741429e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(6.5f), 12),
      static_cast<T>(0.4778085851102157284559772014617919921875e19L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.125)),
      static_cast<T>(0.101884810991335118067599354446661763518e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.25)),
      static_cast<T>(0.1076605959896767217287560924887657165527e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(0.75)),
      static_cast<T>(0.2041755737073560794669901952147483825684e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8060387357327405373376954971672611741429e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12c, static_cast<T>(6.5f)),
      static_cast<T>(0.4778085851102157284559772014617919921875e19L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12a, static_cast<T>(0.125)),
      static_cast<T>(0.101884810991335118067599354446661763518e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12a, static_cast<T>(0.25)),
      static_cast<T>(0.1076605959896767217287560924887657165527e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12a, static_cast<T>(0.75)),
      static_cast<T>(0.2041755737073560794669901952147483825684e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8060387357327405373376954971672611741429e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n12a, static_cast<T>(6.5f)),
      static_cast<T>(0.4778085851102157284559772014617919921875e19L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.125), 12),
      static_cast<T>(0.1150784879306809445407948355732941081442e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.25), 12),
      static_cast<T>(0.1306423839587068869150243699550628662109e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.75), 12),
      static_cast<T>(0.2389007649431414392893202602863311767578e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f), 12),
      static_cast<T>(0.8172456997919903871367065368048367483356e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(6.5f), 12),
      static_cast<T>(0.73509013093879343685534954071044921875e18L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.125)),
      static_cast<T>(0.1150784879306809445407948355732941081442e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.25)),
      static_cast<T>(0.1306423839587068869150243699550628662109e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(0.75)),
      static_cast<T>(0.2389007649431414392893202602863311767578e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8172456997919903871367065368048367483356e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12c, static_cast<T>(6.5f)),
      static_cast<T>(0.73509013093879343685534954071044921875e18L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12a, static_cast<T>(0.125)),
      static_cast<T>(0.1150784879306809445407948355732941081442e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12a, static_cast<T>(0.25)),
      static_cast<T>(0.1306423839587068869150243699550628662109e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12a, static_cast<T>(0.75)),
      static_cast<T>(0.2389007649431414392893202602863311767578e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8172456997919903871367065368048367483356e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n12a, static_cast<T>(6.5f)),
      static_cast<T>(0.73509013093879343685534954071044921875e18L),
      tolerance);
   //
   // Rational functions of order 11
   //
   static const U d12c[12] = { 12, 5, 2, 8, 3, 2, 6, 9, 2, 8, 9, 12 };
   static const boost::array<U, 12> d12a = {{ 12, 5, 2, 8, 3, 2, 6, 9, 2, 8, 9, 12 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.125), 12),
      static_cast<T>(0.9128003783370762743953357962892418132189e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.25), 12),
      static_cast<T>(0.9857041905689267091438933694601440838819e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.75), 12),
      static_cast<T>(0.1248112763387283893598834927961632655902e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(1.0f - 1.0f/64.0f), 12),
      static_cast<T>(0.1227813945781309965073515980672922656926e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(6.5f), 12),
      static_cast<T>(0.5670462630528956417277364302989555872917e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(10247.25f), 12),
      static_cast<T>(0.5000447249679368028702341332079904080375e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.125)),
      static_cast<T>(0.9128003783370762743953357962892418132189e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.25)),
      static_cast<T>(0.9857041905689267091438933694601440838819e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(0.75)),
      static_cast<T>(0.1248112763387283893598834927961632655902e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1227813945781309965073515980672922656926e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(6.5f)),
      static_cast<T>(0.5670462630528956417277364302989555872917e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12c, d12c, static_cast<T>(10247.25f)),
      static_cast<T>(0.5000447249679368028702341332079904080375e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(0.125)),
      static_cast<T>(0.9128003783370762743953357962892418132189e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(0.25)),
      static_cast<T>(0.9857041905689267091438933694601440838819e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(0.75)),
      static_cast<T>(0.1248112763387283893598834927961632655902e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1227813945781309965073515980672922656926e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(6.5f)),
      static_cast<T>(0.5670462630528956417277364302989555872917e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n12a, d12a, static_cast<T>(10247.25f)),
      static_cast<T>(0.5000447249679368028702341332079904080375e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots12(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 12
   //
   static const U n13c[13] = { 4, 11, 7, 1, 1, 1, 8, 11, 10, 12, 8, 2, 1 };
   static const boost::array<U, 13> n13a = {{ 4, 11, 7, 1, 1, 1, 8, 11, 10, 12, 8, 2, 1 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.125), 13),
      static_cast<T>(0.5486639239141368307173252105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.25), 13),
      static_cast<T>(0.7210838854312896728515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.75), 13),
      static_cast<T>(0.22524036943912506103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f), 13),
      static_cast<T>(0.7013317633407797455061737623412208147977e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(6.5), 13),
      static_cast<T>(0.8801602436469970703125e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.125)),
      static_cast<T>(0.5486639239141368307173252105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.25)),
      static_cast<T>(0.7210838854312896728515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(0.75)),
      static_cast<T>(0.22524036943912506103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7013317633407797455061737623412208147977e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13c, static_cast<T>(6.5)),
      static_cast<T>(0.8801602436469970703125e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13a, static_cast<T>(0.125)),
      static_cast<T>(0.5486639239141368307173252105712890625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13a, static_cast<T>(0.25)),
      static_cast<T>(0.7210838854312896728515625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13a, static_cast<T>(0.75)),
      static_cast<T>(0.22524036943912506103515625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7013317633407797455061737623412208147977e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n13a, static_cast<T>(6.5)),
      static_cast<T>(0.8801602436469970703125e10L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.125), 13),
      static_cast<T>(0.4173587859727185607499727379291788731397e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.25), 13),
      static_cast<T>(0.4715104623414053008900737040676176548004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.75), 13),
      static_cast<T>(0.1338397611887558369403450342360883951187e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f), 13),
      static_cast<T>(0.6407202225497595548875186059951665936749e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(6.5f), 13),
      static_cast<T>(0.3403521961549553788009664398431777954102e20L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.125)),
      static_cast<T>(0.4173587859727185607499727379291788731397e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.25)),
      static_cast<T>(0.4715104623414053008900737040676176548004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(0.75)),
      static_cast<T>(0.1338397611887558369403450342360883951187e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6407202225497595548875186059951665936749e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13c, static_cast<T>(6.5f)),
      static_cast<T>(0.3403521961549553788009664398431777954102e20L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13a, static_cast<T>(0.125)),
      static_cast<T>(0.4173587859727185607499727379291788731397e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13a, static_cast<T>(0.25)),
      static_cast<T>(0.4715104623414053008900737040676176548004e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13a, static_cast<T>(0.75)),
      static_cast<T>(0.1338397611887558369403450342360883951187e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6407202225497595548875186059951665936749e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n13a, static_cast<T>(6.5f)),
      static_cast<T>(0.3403521961549553788009664398431777954102e20L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.125), 13),
      static_cast<T>(0.5388702877817484859997819034334309851175e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.25), 13),
      static_cast<T>(0.6860418493656212035602948162704706192017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.75), 13),
      static_cast<T>(0.165119681585007782587126712314784526825e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f), 13),
      static_cast<T>(0.6502554641775335160762093775188993967491e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(6.5f), 13),
      static_cast<T>(0.5236187633153159677245637536048889160156e19L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.125)),
      static_cast<T>(0.5388702877817484859997819034334309851175e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.25)),
      static_cast<T>(0.6860418493656212035602948162704706192017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(0.75)),
      static_cast<T>(0.165119681585007782587126712314784526825e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6502554641775335160762093775188993967491e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13c, static_cast<T>(6.5f)),
      static_cast<T>(0.5236187633153159677245637536048889160156e19L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13a, static_cast<T>(0.125)),
      static_cast<T>(0.5388702877817484859997819034334309851175e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13a, static_cast<T>(0.25)),
      static_cast<T>(0.6860418493656212035602948162704706192017e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13a, static_cast<T>(0.75)),
      static_cast<T>(0.165119681585007782587126712314784526825e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.6502554641775335160762093775188993967491e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n13a, static_cast<T>(6.5f)),
      static_cast<T>(0.5236187633153159677245637536048889160156e19L),
      tolerance);
   //
   // Rational functions of order 12
   //
   static const U d13c[13] = { 4, 7, 1, 6, 11, 4, 9, 11, 1, 10, 1, 11, 12 };
   static const boost::array<U, 13> d13a = {{ 4, 7, 1, 6, 11, 4, 9, 11, 1, 10, 1, 11, 12 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.125), 13),
      static_cast<T>(0.1118537310443419140823936840990235560775e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.25), 13),
      static_cast<T>(0.12106743933208147312270004711488386001e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.75), 13),
      static_cast<T>(0.1042994706832119738384940706309786965245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(1.0f - 1.0f/64.0f), 13),
      static_cast<T>(0.8849937505065081554917467830175954168154e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(6.5f), 13),
      static_cast<T>(0.1125049235146435310211414108934070096386e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(10247.25f), 13),
      static_cast<T>(0.8334214878002966556152610218152238753709e-1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.125)),
      static_cast<T>(0.1118537310443419140823936840990235560775e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.25)),
      static_cast<T>(0.12106743933208147312270004711488386001e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(0.75)),
      static_cast<T>(0.1042994706832119738384940706309786965245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8849937505065081554917467830175954168154e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(6.5f)),
      static_cast<T>(0.1125049235146435310211414108934070096386e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13c, d13c, static_cast<T>(10247.25f)),
      static_cast<T>(0.8334214878002966556152610218152238753709e-1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(0.125)),
      static_cast<T>(0.1118537310443419140823936840990235560775e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(0.25)),
      static_cast<T>(0.12106743933208147312270004711488386001e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(0.75)),
      static_cast<T>(0.1042994706832119738384940706309786965245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8849937505065081554917467830175954168154e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(6.5f)),
      static_cast<T>(0.1125049235146435310211414108934070096386e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n13a, d13a, static_cast<T>(10247.25f)),
      static_cast<T>(0.8334214878002966556152610218152238753709e-1L),
      tolerance);
}
template <class T, class U>
void do_test_spots13(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 13
   //
   static const U n14c[14] = { 5, 5, 3, 5, 12, 8, 10, 5, 5, 9, 2, 10, 3, 3 };
   static const boost::array<U, 14> n14a = {{ 5, 5, 3, 5, 12, 8, 10, 5, 5, 9, 2, 10, 3, 3 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.125), 14),
      static_cast<T>(0.5684855352437807596288621425628662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.25), 14),
      static_cast<T>(0.657317422330379486083984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.75), 14),
      static_cast<T>(0.2256699807941913604736328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f), 14),
      static_cast<T>(0.7710103154409585858207746641830426369713e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(6.5), 14),
      static_cast<T>(0.1372059025011920166015625e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.125)),
      static_cast<T>(0.5684855352437807596288621425628662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.25)),
      static_cast<T>(0.657317422330379486083984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(0.75)),
      static_cast<T>(0.2256699807941913604736328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7710103154409585858207746641830426369713e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14c, static_cast<T>(6.5)),
      static_cast<T>(0.1372059025011920166015625e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14a, static_cast<T>(0.125)),
      static_cast<T>(0.5684855352437807596288621425628662109375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14a, static_cast<T>(0.25)),
      static_cast<T>(0.657317422330379486083984375e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14a, static_cast<T>(0.75)),
      static_cast<T>(0.2256699807941913604736328125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7710103154409585858207746641830426369713e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n14a, static_cast<T>(6.5)),
      static_cast<T>(0.1372059025011920166015625e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.125), 14),
      static_cast<T>(0.5078877218214320312311861835604176527818e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.25), 14),
      static_cast<T>(0.5325630803958699699407475236512254923582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.75), 14),
      static_cast<T>(0.1183906394403697492911931021808413788676e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f), 14),
      static_cast<T>(0.7015764398304385537424317046920004042802e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(6.5f), 14),
      static_cast<T>(0.4205557544065332561152158209905028343201e22L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.125)),
      static_cast<T>(0.5078877218214320312311861835604176527818e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.25)),
      static_cast<T>(0.5325630803958699699407475236512254923582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(0.75)),
      static_cast<T>(0.1183906394403697492911931021808413788676e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7015764398304385537424317046920004042802e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14c, static_cast<T>(6.5f)),
      static_cast<T>(0.4205557544065332561152158209905028343201e22L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14a, static_cast<T>(0.125)),
      static_cast<T>(0.5078877218214320312311861835604176527818e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14a, static_cast<T>(0.25)),
      static_cast<T>(0.5325630803958699699407475236512254923582e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14a, static_cast<T>(0.75)),
      static_cast<T>(0.1183906394403697492911931021808413788676e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7015764398304385537424317046920004042802e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n14a, static_cast<T>(6.5f)),
      static_cast<T>(0.4205557544065332561152158209905028343201e22L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.125), 14),
      static_cast<T>(0.5631017745714562498494894684833412222547e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.25), 14),
      static_cast<T>(0.6302523215834798797629900946049019694328e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.75), 14),
      static_cast<T>(0.1411875192538263323882574695744551718235e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f), 14),
      static_cast<T>(0.7119189230023502768177083984172702519672e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(6.5f), 14),
      static_cast<T>(0.6470088529331280863353320322930812835693e21L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.125)),
      static_cast<T>(0.5631017745714562498494894684833412222547e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.25)),
      static_cast<T>(0.6302523215834798797629900946049019694328e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(0.75)),
      static_cast<T>(0.1411875192538263323882574695744551718235e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7119189230023502768177083984172702519672e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14c, static_cast<T>(6.5f)),
      static_cast<T>(0.6470088529331280863353320322930812835693e21L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14a, static_cast<T>(0.125)),
      static_cast<T>(0.5631017745714562498494894684833412222547e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14a, static_cast<T>(0.25)),
      static_cast<T>(0.6302523215834798797629900946049019694328e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14a, static_cast<T>(0.75)),
      static_cast<T>(0.1411875192538263323882574695744551718235e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7119189230023502768177083984172702519672e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n14a, static_cast<T>(6.5f)),
      static_cast<T>(0.6470088529331280863353320322930812835693e21L),
      tolerance);
   //
   // Rational functions of order 13
   //
   static const U d14c[14] = { 1, 2, 1, 8, 5, 8, 2, 11, 3, 6, 5, 9, 7, 10 };
   static const boost::array<U, 14> d14a = {{ 1, 2, 1, 8, 5, 8, 2, 11, 3, 6, 5, 9, 7, 10 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.125), 14),
      static_cast<T>(0.4431848049037056640776200482297774574883e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.25), 14),
      static_cast<T>(0.3830343514018902583833597521450902906851e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.75), 14),
      static_cast<T>(0.1657621200711495617835320787645038062003e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(1.0f - 1.0f/64.0f), 14),
      static_cast<T>(0.1117985091800992684003226640313705018678e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(6.5f), 14),
      static_cast<T>(0.3280672161259396135368350556553925507681e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(10247.25f), 14),
      static_cast<T>(0.3000087891957249835881866421118445480875e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.125)),
      static_cast<T>(0.4431848049037056640776200482297774574883e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.25)),
      static_cast<T>(0.3830343514018902583833597521450902906851e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(0.75)),
      static_cast<T>(0.1657621200711495617835320787645038062003e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1117985091800992684003226640313705018678e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(6.5f)),
      static_cast<T>(0.3280672161259396135368350556553925507681e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14c, d14c, static_cast<T>(10247.25f)),
      static_cast<T>(0.3000087891957249835881866421118445480875e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(0.125)),
      static_cast<T>(0.4431848049037056640776200482297774574883e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(0.25)),
      static_cast<T>(0.3830343514018902583833597521450902906851e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(0.75)),
      static_cast<T>(0.1657621200711495617835320787645038062003e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1117985091800992684003226640313705018678e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(6.5f)),
      static_cast<T>(0.3280672161259396135368350556553925507681e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n14a, d14a, static_cast<T>(10247.25f)),
      static_cast<T>(0.3000087891957249835881866421118445480875e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots14(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 14
   //
   static const U n15c[15] = { 6, 2, 10, 2, 4, 11, 4, 6, 5, 5, 10, 12, 8, 6, 2 };
   static const boost::array<U, 15> n15a = {{ 6, 2, 10, 2, 4, 11, 4, 6, 5, 5, 10, 12, 8, 6, 2 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.125), 15),
      static_cast<T>(0.6411486971785507193999364972114562988281e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.25), 15),
      static_cast<T>(0.7184068299829959869384765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.75), 15),
      static_cast<T>(0.21735079027712345123291015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f), 15),
      static_cast<T>(0.8299984286646360949452954853655163780233e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(6.5), 15),
      static_cast<T>(0.7599432765217742919921875e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.125)),
      static_cast<T>(0.6411486971785507193999364972114562988281e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.25)),
      static_cast<T>(0.7184068299829959869384765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(0.75)),
      static_cast<T>(0.21735079027712345123291015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8299984286646360949452954853655163780233e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15c, static_cast<T>(6.5)),
      static_cast<T>(0.7599432765217742919921875e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15a, static_cast<T>(0.125)),
      static_cast<T>(0.6411486971785507193999364972114562988281e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15a, static_cast<T>(0.25)),
      static_cast<T>(0.7184068299829959869384765625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15a, static_cast<T>(0.75)),
      static_cast<T>(0.21735079027712345123291015625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8299984286646360949452954853655163780233e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n15a, static_cast<T>(6.5)),
      static_cast<T>(0.7599432765217742919921875e12L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.125), 15),
      static_cast<T>(0.603369928436724862526100235763586452224e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.25), 15),
      static_cast<T>(0.6164622568840771005271861326946236658841e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.75), 15),
      static_cast<T>(0.1204200081304656236302896843426424311474e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f), 15),
      static_cast<T>(0.7438435296991670658675730793779505088179e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(6.5f), 15),
      static_cast<T>(0.123975649339443727300884144392229616642e24L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.125)),
      static_cast<T>(0.603369928436724862526100235763586452224e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.25)),
      static_cast<T>(0.6164622568840771005271861326946236658841e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(0.75)),
      static_cast<T>(0.1204200081304656236302896843426424311474e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7438435296991670658675730793779505088179e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15c, static_cast<T>(6.5f)),
      static_cast<T>(0.123975649339443727300884144392229616642e24L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15a, static_cast<T>(0.125)),
      static_cast<T>(0.603369928436724862526100235763586452224e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15a, static_cast<T>(0.25)),
      static_cast<T>(0.6164622568840771005271861326946236658841e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15a, static_cast<T>(0.75)),
      static_cast<T>(0.1204200081304656236302896843426424311474e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7438435296991670658675730793779505088179e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n15a, static_cast<T>(6.5f)),
      static_cast<T>(0.123975649339443727300884144392229616642e24L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.125), 15),
      static_cast<T>(0.6269594274937989002088018861086916177919e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.25), 15),
      static_cast<T>(0.6658490275363084021087445307784946635365e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.75), 15),
      static_cast<T>(0.1405600108406208315070529124568565748632e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f), 15),
      static_cast<T>(0.7546981889007411462781694774633148026087e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(6.5f), 15),
      static_cast<T>(0.1907317682145288112321802221418917179108e23L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.125)),
      static_cast<T>(0.6269594274937989002088018861086916177919e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.25)),
      static_cast<T>(0.6658490275363084021087445307784946635365e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(0.75)),
      static_cast<T>(0.1405600108406208315070529124568565748632e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7546981889007411462781694774633148026087e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15c, static_cast<T>(6.5f)),
      static_cast<T>(0.1907317682145288112321802221418917179108e23L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15a, static_cast<T>(0.125)),
      static_cast<T>(0.6269594274937989002088018861086916177919e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15a, static_cast<T>(0.25)),
      static_cast<T>(0.6658490275363084021087445307784946635365e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15a, static_cast<T>(0.75)),
      static_cast<T>(0.1405600108406208315070529124568565748632e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7546981889007411462781694774633148026087e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n15a, static_cast<T>(6.5f)),
      static_cast<T>(0.1907317682145288112321802221418917179108e23L),
      tolerance);
   //
   // Rational functions of order 14
   //
   static const U d15c[15] = { 7, 10, 7, 1, 10, 11, 11, 10, 7, 1, 10, 9, 8, 2, 3 };
   static const boost::array<U, 15> d15a = {{ 7, 10, 7, 1, 10, 11, 11, 10, 7, 1, 10, 9, 8, 2, 3 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.125), 15),
      static_cast<T>(0.7665435387801205084357966990293734431712e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.25), 15),
      static_cast<T>(0.7179510450452743437979498898508621162489e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.75), 15),
      static_cast<T>(0.7245051156202447814210400449367214490615e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(1.0f - 1.0f/64.0f), 15),
      static_cast<T>(0.8564567127146107292325365412425895512358e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(6.5f), 15),
      static_cast<T>(0.8943937154363365123476971638815814742919e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(10247.25f), 15),
      static_cast<T>(0.6668184674989563469735607034881326982176e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.125)),
      static_cast<T>(0.7665435387801205084357966990293734431712e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.25)),
      static_cast<T>(0.7179510450452743437979498898508621162489e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(0.75)),
      static_cast<T>(0.7245051156202447814210400449367214490615e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8564567127146107292325365412425895512358e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(6.5f)),
      static_cast<T>(0.8943937154363365123476971638815814742919e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15c, d15c, static_cast<T>(10247.25f)),
      static_cast<T>(0.6668184674989563469735607034881326982176e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(0.125)),
      static_cast<T>(0.7665435387801205084357966990293734431712e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(0.25)),
      static_cast<T>(0.7179510450452743437979498898508621162489e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(0.75)),
      static_cast<T>(0.7245051156202447814210400449367214490615e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8564567127146107292325365412425895512358e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(6.5f)),
      static_cast<T>(0.8943937154363365123476971638815814742919e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n15a, d15a, static_cast<T>(10247.25f)),
      static_cast<T>(0.6668184674989563469735607034881326982176e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots15(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 15
   //
   static const U n16c[16] = { 5, 3, 4, 6, 7, 9, 7, 6, 12, 12, 7, 12, 10, 3, 6, 2 };
   static const boost::array<U, 16> n16a = {{ 5, 3, 4, 6, 7, 9, 7, 6, 12, 12, 7, 12, 10, 3, 6, 2 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.125), 16),
      static_cast<T>(0.545123276921327715172083117067813873291e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.25), 16),
      static_cast<T>(0.613219709135591983795166015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.75), 16),
      static_cast<T>(0.2195364624075591564178466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f), 16),
      static_cast<T>(0.9836568924934260045838041177111855604603e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(6.5), 16),
      static_cast<T>(0.474557924297607818603515625e13L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.125)),
      static_cast<T>(0.545123276921327715172083117067813873291e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.25)),
      static_cast<T>(0.613219709135591983795166015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(0.75)),
      static_cast<T>(0.2195364624075591564178466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9836568924934260045838041177111855604603e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16c, static_cast<T>(6.5)),
      static_cast<T>(0.474557924297607818603515625e13L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16a, static_cast<T>(0.125)),
      static_cast<T>(0.545123276921327715172083117067813873291e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16a, static_cast<T>(0.25)),
      static_cast<T>(0.613219709135591983795166015625e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16a, static_cast<T>(0.75)),
      static_cast<T>(0.2195364624075591564178466796875e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9836568924934260045838041177111855604603e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n16a, static_cast<T>(6.5)),
      static_cast<T>(0.474557924297607818603515625e13L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.125), 16),
      static_cast<T>(0.5047874876401281302859159688868887179229e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.25), 16),
      static_cast<T>(0.5204705680902215943553490440365294489311e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.75), 16),
      static_cast<T>(0.1080276729363941941482185615797106947866e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f), 16),
      static_cast<T>(0.8750139513999183433264859554731006794881e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(6.5f), 16),
      static_cast<T>(0.5231073382834479254225049897383445873857e25L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.125)),
      static_cast<T>(0.5047874876401281302859159688868887179229e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.25)),
      static_cast<T>(0.5204705680902215943553490440365294489311e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(0.75)),
      static_cast<T>(0.1080276729363941941482185615797106947866e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8750139513999183433264859554731006794881e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16c, static_cast<T>(6.5f)),
      static_cast<T>(0.5231073382834479254225049897383445873857e25L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16a, static_cast<T>(0.125)),
      static_cast<T>(0.5047874876401281302859159688868887179229e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16a, static_cast<T>(0.25)),
      static_cast<T>(0.5204705680902215943553490440365294489311e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16a, static_cast<T>(0.75)),
      static_cast<T>(0.1080276729363941941482185615797106947866e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8750139513999183433264859554731006794881e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n16a, static_cast<T>(6.5f)),
      static_cast<T>(0.5231073382834479254225049897383445873857e25L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.125), 16),
      static_cast<T>(0.5382999011210250422873277510951097433836e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.25), 16),
      static_cast<T>(0.5818822723608863774213961761461177957244e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.75), 16),
      static_cast<T>(0.1273702305818589255309580821062809263822e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f), 16),
      static_cast<T>(0.8881094109459487932205571611155308490038e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(6.5f), 16),
      static_cast<T>(0.8047805204360737314192426765205301344395e24L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.125)),
      static_cast<T>(0.5382999011210250422873277510951097433836e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.25)),
      static_cast<T>(0.5818822723608863774213961761461177957244e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(0.75)),
      static_cast<T>(0.1273702305818589255309580821062809263822e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8881094109459487932205571611155308490038e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16c, static_cast<T>(6.5f)),
      static_cast<T>(0.8047805204360737314192426765205301344395e24L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16a, static_cast<T>(0.125)),
      static_cast<T>(0.5382999011210250422873277510951097433836e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16a, static_cast<T>(0.25)),
      static_cast<T>(0.5818822723608863774213961761461177957244e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16a, static_cast<T>(0.75)),
      static_cast<T>(0.1273702305818589255309580821062809263822e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8881094109459487932205571611155308490038e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n16a, static_cast<T>(6.5f)),
      static_cast<T>(0.8047805204360737314192426765205301344395e24L),
      tolerance);
   //
   // Rational functions of order 15
   //
   static const U d16c[16] = { 9, 5, 2, 4, 4, 12, 11, 5, 1, 4, 1, 7, 9, 6, 6, 10 };
   static const boost::array<U, 16> d16a = {{ 9, 5, 2, 4, 4, 12, 11, 5, 1, 4, 1, 7, 9, 6, 6, 10 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.125), 16),
      static_cast<T>(0.5639916333371058636233240074655101514789e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.25), 16),
      static_cast<T>(0.5858114501425065583332351775671440891069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.75), 16),
      static_cast<T>(0.9254727283150680371198227978127749853934e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(1.0f - 1.0f/64.0f), 16),
      static_cast<T>(0.1155192269792597191440811772688208898465e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(6.5f), 16),
      static_cast<T>(0.2736469218296852258039959401695110274611e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(10247.25f), 16),
      static_cast<T>(0.2000468408076577399196121638600795513365e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.125)),
      static_cast<T>(0.5639916333371058636233240074655101514789e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.25)),
      static_cast<T>(0.5858114501425065583332351775671440891069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(0.75)),
      static_cast<T>(0.9254727283150680371198227978127749853934e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1155192269792597191440811772688208898465e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(6.5f)),
      static_cast<T>(0.2736469218296852258039959401695110274611e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16c, d16c, static_cast<T>(10247.25f)),
      static_cast<T>(0.2000468408076577399196121638600795513365e0L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(0.125)),
      static_cast<T>(0.5639916333371058636233240074655101514789e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(0.25)),
      static_cast<T>(0.5858114501425065583332351775671440891069e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(0.75)),
      static_cast<T>(0.9254727283150680371198227978127749853934e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.1155192269792597191440811772688208898465e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(6.5f)),
      static_cast<T>(0.2736469218296852258039959401695110274611e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n16a, d16a, static_cast<T>(10247.25f)),
      static_cast<T>(0.2000468408076577399196121638600795513365e0L),
      tolerance);
}

template <class T, class U>
void do_test_spots16(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 16
   //
   static const U n17c[17] = { 9, 6, 11, 2, 9, 10, 6, 3, 3, 3, 4, 9, 10, 2, 3, 2, 2 };
   static const boost::array<U, 17> n17a = {{ 9, 6, 11, 2, 9, 10, 6, 3, 3, 3, 4, 9, 10, 2, 3, 2, 2 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.125), 17),
      static_cast<T>(0.9928308216189925872185995103791356086731e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.25), 17),
      static_cast<T>(0.112653836444951593875885009765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.75), 17),
      static_cast<T>(0.288157429718412458896636962890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f), 17),
      static_cast<T>(0.8499409476622319741012411965028222548509e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(6.5), 17),
      static_cast<T>(0.24291309657542805938720703125e14L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.125)),
      static_cast<T>(0.9928308216189925872185995103791356086731e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.25)),
      static_cast<T>(0.112653836444951593875885009765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(0.75)),
      static_cast<T>(0.288157429718412458896636962890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8499409476622319741012411965028222548509e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17c, static_cast<T>(6.5)),
      static_cast<T>(0.24291309657542805938720703125e14L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17a, static_cast<T>(0.125)),
      static_cast<T>(0.9928308216189925872185995103791356086731e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17a, static_cast<T>(0.25)),
      static_cast<T>(0.112653836444951593875885009765625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17a, static_cast<T>(0.75)),
      static_cast<T>(0.288157429718412458896636962890625e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8499409476622319741012411965028222548509e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n17a, static_cast<T>(6.5)),
      static_cast<T>(0.24291309657542805938720703125e14L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.125), 17),
      static_cast<T>(0.9096443722112564415334975111453014308977e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.25), 17),
      static_cast<T>(0.9418604266644799371155545586464796770088e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.75), 17),
      static_cast<T>(0.1800748238050061998856161277204890325265e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f), 17),
      static_cast<T>(0.7725414891276907880696219594586980069116e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(6.5f), 17),
      static_cast<T>(0.2112416072820759278619692697965107844211e27L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.125)),
      static_cast<T>(0.9096443722112564415334975111453014308977e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.25)),
      static_cast<T>(0.9418604266644799371155545586464796770088e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(0.75)),
      static_cast<T>(0.1800748238050061998856161277204890325265e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7725414891276907880696219594586980069116e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17c, static_cast<T>(6.5f)),
      static_cast<T>(0.2112416072820759278619692697965107844211e27L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17a, static_cast<T>(0.125)),
      static_cast<T>(0.9096443722112564415334975111453014308977e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17a, static_cast<T>(0.25)),
      static_cast<T>(0.9418604266644799371155545586464796770088e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17a, static_cast<T>(0.75)),
      static_cast<T>(0.1800748238050061998856161277204890325265e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7725414891276907880696219594586980069116e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n17a, static_cast<T>(6.5f)),
      static_cast<T>(0.2112416072820759278619692697965107844211e27L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.125), 17),
      static_cast<T>(0.9771549776900515322679800891624114471814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.25), 17),
      static_cast<T>(0.1067441706657919748462218234585918708035e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.75), 17),
      static_cast<T>(0.210099765073341599847488170293985376702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f), 17),
      static_cast<T>(0.7833754810186065148643778635770900387673e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(6.5f), 17),
      static_cast<T>(0.3249870881262706582491835681484781298786e26L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.125)),
      static_cast<T>(0.9771549776900515322679800891624114471814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.25)),
      static_cast<T>(0.1067441706657919748462218234585918708035e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(0.75)),
      static_cast<T>(0.210099765073341599847488170293985376702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7833754810186065148643778635770900387673e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17c, static_cast<T>(6.5f)),
      static_cast<T>(0.3249870881262706582491835681484781298786e26L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17a, static_cast<T>(0.125)),
      static_cast<T>(0.9771549776900515322679800891624114471814e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17a, static_cast<T>(0.25)),
      static_cast<T>(0.1067441706657919748462218234585918708035e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17a, static_cast<T>(0.75)),
      static_cast<T>(0.210099765073341599847488170293985376702e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7833754810186065148643778635770900387673e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n17a, static_cast<T>(6.5f)),
      static_cast<T>(0.3249870881262706582491835681484781298786e26L),
      tolerance);
   //
   // Rational functions of order 16
   //
   static const U d17c[17] = { 7, 12, 3, 11, 10, 2, 5, 10, 4, 11, 10, 6, 2, 12, 1, 2, 1 };
   static const boost::array<U, 17> d17a = {{ 7, 12, 3, 11, 10, 2, 5, 10, 4, 11, 10, 6, 2, 12, 1, 2, 1 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.125), 17),
      static_cast<T>(0.1158375951946763080209403673166347533318e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.25), 17),
      static_cast<T>(0.1082966720285644924561131930917359028989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.75), 17),
      static_cast<T>(0.9405394059495731884564238556014726773139e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(1.0f - 1.0f/64.0f), 17),
      static_cast<T>(0.8654376623250381987950547557864849720673e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(6.5f), 17),
      static_cast<T>(0.1737601600028386310221746538845672721834e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(10247.25f), 17),
      static_cast<T>(0.1999804873273333726353229695741031917586e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.125)),
      static_cast<T>(0.1158375951946763080209403673166347533318e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.25)),
      static_cast<T>(0.1082966720285644924561131930917359028989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(0.75)),
      static_cast<T>(0.9405394059495731884564238556014726773139e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8654376623250381987950547557864849720673e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(6.5f)),
      static_cast<T>(0.1737601600028386310221746538845672721834e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17c, d17c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1999804873273333726353229695741031917586e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(0.125)),
      static_cast<T>(0.1158375951946763080209403673166347533318e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(0.25)),
      static_cast<T>(0.1082966720285644924561131930917359028989e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(0.75)),
      static_cast<T>(0.9405394059495731884564238556014726773139e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8654376623250381987950547557864849720673e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(6.5f)),
      static_cast<T>(0.1737601600028386310221746538845672721834e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n17a, d17a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1999804873273333726353229695741031917586e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots17(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 17
   //
   static const U n18c[18] = { 6, 5, 7, 8, 12, 9, 4, 1, 3, 5, 1, 5, 12, 8, 4, 6, 1, 10 };
   static const boost::array<U, 18> n18a = {{ 6, 5, 7, 8, 12, 9, 4, 1, 3, 5, 1, 5, 12, 8, 4, 6, 1, 10 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.125), 18),
      static_cast<T>(0.6753220299099845114199069939786568284035e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.25), 18),
      static_cast<T>(0.7869269511546008288860321044921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.75), 18),
      static_cast<T>(0.25590585466590709984302520751953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f), 18),
      static_cast<T>(0.9432688198514753127308115619023015607828e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(6.5), 18),
      static_cast<T>(0.680830157865510950897216796875e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.125)),
      static_cast<T>(0.6753220299099845114199069939786568284035e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.25)),
      static_cast<T>(0.7869269511546008288860321044921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(0.75)),
      static_cast<T>(0.25590585466590709984302520751953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9432688198514753127308115619023015607828e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18c, static_cast<T>(6.5)),
      static_cast<T>(0.680830157865510950897216796875e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18a, static_cast<T>(0.125)),
      static_cast<T>(0.6753220299099845114199069939786568284035e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18a, static_cast<T>(0.25)),
      static_cast<T>(0.7869269511546008288860321044921875e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18a, static_cast<T>(0.75)),
      static_cast<T>(0.25590585466590709984302520751953125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9432688198514753127308115619023015607828e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n18a, static_cast<T>(6.5)),
      static_cast<T>(0.680830157865510950897216796875e15L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.125), 18),
      static_cast<T>(0.6079865225649211447450201991749585071453e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.25), 18),
      static_cast<T>(0.6341988806453956744898332268528529098717e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.75), 18),
      static_cast<T>(0.1439378129034890784564674906867431936064e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f), 18),
      static_cast<T>(0.8374908316692411299816073455081075151876e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(6.5f), 18),
      static_cast<T>(0.4367459075155664096657376949959960997093e29L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.125)),
      static_cast<T>(0.6079865225649211447450201991749585071453e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.25)),
      static_cast<T>(0.6341988806453956744898332268528529098717e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(0.75)),
      static_cast<T>(0.1439378129034890784564674906867431936064e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8374908316692411299816073455081075151876e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18c, static_cast<T>(6.5f)),
      static_cast<T>(0.4367459075155664096657376949959960997093e29L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18a, static_cast<T>(0.125)),
      static_cast<T>(0.6079865225649211447450201991749585071453e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18a, static_cast<T>(0.25)),
      static_cast<T>(0.6341988806453956744898332268528529098717e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18a, static_cast<T>(0.75)),
      static_cast<T>(0.1439378129034890784564674906867431936064e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8374908316692411299816073455081075151876e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n18a, static_cast<T>(6.5f)),
      static_cast<T>(0.4367459075155664096657376949959960997093e29L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.125), 18),
      static_cast<T>(0.6638921805193691579601615933996680571626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.25), 18),
      static_cast<T>(0.7367955225815826979593329074114116394867e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.75), 18),
      static_cast<T>(0.1719170838713187712752899875823242581419e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f), 18),
      static_cast<T>(0.8498319559814513066479820652780774757461e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(6.5f), 18),
      static_cast<T>(0.6719167807931790917934426081938401533989e28L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.125)),
      static_cast<T>(0.6638921805193691579601615933996680571626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.25)),
      static_cast<T>(0.7367955225815826979593329074114116394867e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(0.75)),
      static_cast<T>(0.1719170838713187712752899875823242581419e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8498319559814513066479820652780774757461e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18c, static_cast<T>(6.5f)),
      static_cast<T>(0.6719167807931790917934426081938401533989e28L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18a, static_cast<T>(0.125)),
      static_cast<T>(0.6638921805193691579601615933996680571626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18a, static_cast<T>(0.25)),
      static_cast<T>(0.7367955225815826979593329074114116394867e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18a, static_cast<T>(0.75)),
      static_cast<T>(0.1719170838713187712752899875823242581419e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8498319559814513066479820652780774757461e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n18a, static_cast<T>(6.5f)),
      static_cast<T>(0.6719167807931790917934426081938401533989e28L),
      tolerance);
   //
   // Rational functions of order 17
   //
   static const U d18c[18] = { 6, 2, 11, 2, 12, 4, 1, 5, 7, 12, 5, 7, 5, 7, 5, 7, 2, 9 };
   static const boost::array<U, 18> d18a = {{ 6, 2, 11, 2, 12, 4, 1, 5, 7, 12, 5, 7, 5, 7, 5, 7, 2, 9 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.125), 18),
      static_cast<T>(0.1050457095586493221315235501384188635425e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.25), 18),
      static_cast<T>(0.1082394733850278225180153358339433604182e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.75), 18),
      static_cast<T>(0.1117650157149023587583496399377426188558e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(1.0f - 1.0f/64.0f), 18),
      static_cast<T>(0.9882013696252427983431096654690843223851e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(6.5f), 18),
      static_cast<T>(0.1086348665080997883580887205673821889245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(10247.25f), 18),
      static_cast<T>(0.1111097856940394090609547290690047016487e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.125)),
      static_cast<T>(0.1050457095586493221315235501384188635425e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.25)),
      static_cast<T>(0.1082394733850278225180153358339433604182e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(0.75)),
      static_cast<T>(0.1117650157149023587583496399377426188558e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9882013696252427983431096654690843223851e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(6.5f)),
      static_cast<T>(0.1086348665080997883580887205673821889245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18c, d18c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1111097856940394090609547290690047016487e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(0.125)),
      static_cast<T>(0.1050457095586493221315235501384188635425e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(0.25)),
      static_cast<T>(0.1082394733850278225180153358339433604182e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(0.75)),
      static_cast<T>(0.1117650157149023587583496399377426188558e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.9882013696252427983431096654690843223851e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(6.5f)),
      static_cast<T>(0.1086348665080997883580887205673821889245e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n18a, d18a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1111097856940394090609547290690047016487e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots18(T, U)
{
   //
   // Tolerance is 4 eps expressed as a persentage:
   //
   T tolerance = boost::math::tools::epsilon<T>() * 4 * 100;

   //
   // Polynomials of order 18
   //
   static const U n19c[19] = { 7, 2, 4, 2, 4, 3, 9, 1, 9, 3, 7, 2, 10, 4, 2, 5, 11, 3, 9 };
   static const boost::array<U, 19> n19a = {{ 7, 2, 4, 2, 4, 3, 9, 1, 9, 3, 7, 2, 10, 4, 2, 5, 11, 3, 9 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.125), 19),
      static_cast<T>(0.7317509740045990140888676478425622917712e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.25), 19),
      static_cast<T>(0.7802219584656995721161365509033203125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.75), 19),
      static_cast<T>(0.17609299321266007609665393829345703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f), 19),
      static_cast<T>(0.8327888521934174049333133442289010778727e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(6.5), 19),
      static_cast<T>(0.4179028813935817562572479248046875e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.125)),
      static_cast<T>(0.7317509740045990140888676478425622917712e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.25)),
      static_cast<T>(0.7802219584656995721161365509033203125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(0.75)),
      static_cast<T>(0.17609299321266007609665393829345703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8327888521934174049333133442289010778727e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19c, static_cast<T>(6.5)),
      static_cast<T>(0.4179028813935817562572479248046875e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19a, static_cast<T>(0.125)),
      static_cast<T>(0.7317509740045990140888676478425622917712e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19a, static_cast<T>(0.25)),
      static_cast<T>(0.7802219584656995721161365509033203125e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19a, static_cast<T>(0.75)),
      static_cast<T>(0.17609299321266007609665393829345703125e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8327888521934174049333133442289010778727e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_polynomial(n19a, static_cast<T>(6.5)),
      static_cast<T>(0.4179028813935817562572479248046875e16L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.125), 19),
      static_cast<T>(0.7032234433238304833197011488540991991734e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.25), 19),
      static_cast<T>(0.7141177719741940632063571816842174888595e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.75), 19),
      static_cast<T>(0.107671418972012709727788225142148040292e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f), 19),
      static_cast<T>(0.7205306905267900876445365429703520869133e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(6.5f), 19),
      static_cast<T>(0.1670453627683043936397442394984734181614e31L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.125)),
      static_cast<T>(0.7032234433238304833197011488540991991734e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.25)),
      static_cast<T>(0.7141177719741940632063571816842174888595e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(0.75)),
      static_cast<T>(0.107671418972012709727788225142148040292e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7205306905267900876445365429703520869133e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19c, static_cast<T>(6.5f)),
      static_cast<T>(0.1670453627683043936397442394984734181614e31L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19a, static_cast<T>(0.125)),
      static_cast<T>(0.7032234433238304833197011488540991991734e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19a, static_cast<T>(0.25)),
      static_cast<T>(0.7141177719741940632063571816842174888595e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19a, static_cast<T>(0.75)),
      static_cast<T>(0.107671418972012709727788225142148040292e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7205306905267900876445365429703520869133e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_even_polynomial(n19a, static_cast<T>(6.5f)),
      static_cast<T>(0.1670453627683043936397442394984734181614e31L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.125), 19),
      static_cast<T>(0.7257875465906438665576091908327935933871e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.25), 19),
      static_cast<T>(0.7564710878967762528254287267368699554382e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.75), 19),
      static_cast<T>(0.120228558629350279637050966856197387056e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f), 19),
      static_cast<T>(0.7308565745034058033214339484143259295627e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(6.5f), 19),
      static_cast<T>(0.2569928657973913748303757530804975664022e30L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.125)),
      static_cast<T>(0.7257875465906438665576091908327935933871e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.25)),
      static_cast<T>(0.7564710878967762528254287267368699554382e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(0.75)),
      static_cast<T>(0.120228558629350279637050966856197387056e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7308565745034058033214339484143259295627e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19c, static_cast<T>(6.5f)),
      static_cast<T>(0.2569928657973913748303757530804975664022e30L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19a, static_cast<T>(0.125)),
      static_cast<T>(0.7257875465906438665576091908327935933871e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19a, static_cast<T>(0.25)),
      static_cast<T>(0.7564710878967762528254287267368699554382e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19a, static_cast<T>(0.75)),
      static_cast<T>(0.120228558629350279637050966856197387056e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.7308565745034058033214339484143259295627e2L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_odd_polynomial(n19a, static_cast<T>(6.5f)),
      static_cast<T>(0.2569928657973913748303757530804975664022e30L),
      tolerance);
   //
   // Rational functions of order 18
   //
   static const U d19c[19] = { 3, 2, 3, 3, 10, 6, 2, 6, 9, 8, 8, 10, 5, 7, 6, 4, 6, 9, 7 };
   static const boost::array<U, 19> d19a = {{ 3, 2, 3, 3, 10, 6, 2, 6, 9, 8, 8, 10, 5, 7, 6, 4, 6, 9, 7 }};
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.125), 19),
      static_cast<T>(0.2213824709533496994632324982010106870288e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.25), 19),
      static_cast<T>(0.2063899260318829004588039751781791436716e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.75), 19),
      static_cast<T>(0.1085337280046506127330919833145641152667e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(1.0f - 1.0f/64.0f), 19),
      static_cast<T>(0.8530644127293554041631338796773621506306e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(6.5f), 19),
      static_cast<T>(0.1140014406058131768268407547610620204626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(10247.25f), 19),
      static_cast<T>(0.1285594810697167354761598106316532892359e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.125)),
      static_cast<T>(0.2213824709533496994632324982010106870288e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.25)),
      static_cast<T>(0.2063899260318829004588039751781791436716e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(0.75)),
      static_cast<T>(0.1085337280046506127330919833145641152667e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8530644127293554041631338796773621506306e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(6.5f)),
      static_cast<T>(0.1140014406058131768268407547610620204626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19c, d19c, static_cast<T>(10247.25f)),
      static_cast<T>(0.1285594810697167354761598106316532892359e1L),
      tolerance);

   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(0.125)),
      static_cast<T>(0.2213824709533496994632324982010106870288e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(0.25)),
      static_cast<T>(0.2063899260318829004588039751781791436716e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(0.75)),
      static_cast<T>(0.1085337280046506127330919833145641152667e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(1.0f - 1.0f/64.0f)),
      static_cast<T>(0.8530644127293554041631338796773621506306e0L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(6.5f)),
      static_cast<T>(0.1140014406058131768268407547610620204626e1L),
      tolerance);
   BOOST_CHECK_CLOSE(
      boost::math::tools::evaluate_rational(n19a, d19a, static_cast<T>(10247.25f)),
      static_cast<T>(0.1285594810697167354761598106316532892359e1L),
      tolerance);
}

template <class T, class U>
void do_test_spots(T t, U u)
{
   do_test_spots1(t, u);
   do_test_spots2(t, u);
   do_test_spots3(t, u);
   do_test_spots4(t, u);
   do_test_spots5(t, u);
   do_test_spots6(t, u);
   do_test_spots7(t, u);
   do_test_spots8(t, u);
   do_test_spots9(t, u);
   do_test_spots10(t, u);
   do_test_spots11(t, u);
   do_test_spots12(t, u);
   do_test_spots13(t, u);
   do_test_spots14(t, u);
   do_test_spots15(t, u);
   do_test_spots16(t, u);
   do_test_spots17(t, u);
   do_test_spots18(t, u);
}

#endif
